package ru.yandex.matrixnet;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class BasicMatrixnetModel implements MatrixnetModel {
    private static final int DATA_IDX_INC = 64;
    private static final int INDICIES_IDX_INC = 6;
    private static final int INDICIES_ITERS = 5;

    private static final String END_BRACE = "};";

    private static final VarItem EMPTY_VAR_ITEM = new VarItem();

    private static final Pattern VARS_DECLARE_P
        = Pattern.compile("bool vars\\[(\\d+)\\];");
    private static final Pattern VARS_SET_P = Pattern.compile(
        "vars\\[(\\d+)\\] = fFactorInt\\[(\\d+)\\] > (\\d+) \\? 1 : 0;"
            + " // ([-+]?\\d*\\.?\\d+([eE][-+]?\\d+)?)");
    private static final Pattern ITERATIONS_P = Pattern.compile(
        "for \\(int z = 0; z < (\\d+); \\+\\+z\\) \\{");
    private static final Pattern SCORE_WEIGHT_P = Pattern.compile(
        "double res = (\\d+[\\.]\\d+) \\+ "
            + "resInt \\* ([-+]?\\d*\\.?\\d+([eE][-+]?\\d+)?);");

    private static final int PARSING_INDICIES = 1;
    private static final int PARSING_DATA = 2;

    private int[] indicies;
    private int[] data;
    private int varsNum = -1;
    private int iterations = -1;
    private double scoreOffset = -1;
    private double scoreMultiplier = -1;
    private List<VarItem> varItems;
    private String name;

    public BasicMatrixnetModel(final ImmutableMatrixnetModelConfig config)
        throws MatrixnetModelParseException
    {
        if (config.file() != null) {
            try (FileInputStream fin = new FileInputStream(config.file())) {
                load(fin);
            } catch (IOException io) {
                throw new MatrixnetModelParseException(io);
            }
        } else {
            load(config.content());
        }

        this.name = config.name();
    }

    @Override
    public String name() {
        return name;
    }

    // CSOFF: MagicNumber
    private void load(final InputStream is)
        throws MatrixnetModelParseException
    {
        Scanner scanner = new Scanner(is, "utf-8");
        int status = 0;

        List<int[]> indiciesList = new ArrayList<>();
        List<int[]> dataList = new ArrayList<>();

        int lineNum = 0;
        while (scanner.hasNextLine()) {
            lineNum += 1;
            String line = scanner.nextLine().trim();

            if (line.equals(
                "static unsigned short GeneratedCompactIndicesTbl[] = {"))
            {
                status = PARSING_INDICIES;
                continue;
            }

            if (line.equals("static int GeneratedDataTbl[] = {")) {
                status = PARSING_DATA;
                continue;
            }

            Matcher matcher = VARS_DECLARE_P.matcher(line);
            if (matcher.matches()) {
                if (varsNum != -1 || varItems != null) {
                    throw new MatrixnetModelParseException(
                        "Two times varsNum were parsed",
                        lineNum);
                }

                if (matcher.groupCount() != 1) {
                    throw new MatrixnetModelParseException(
                        "Invalid vars declare string",
                        lineNum);
                }

                varsNum = Integer.parseInt(matcher.group(1));
                varItems = new ArrayList<>(varsNum);
                for (int i = 0; i < varsNum; i++) {
                    varItems.add(EMPTY_VAR_ITEM);
                }
            }

            matcher = VARS_SET_P.matcher(line);
            if (matcher.matches()) {
                if (matcher.groupCount() != 5) {
                    throw new MatrixnetModelParseException(
                        "Invalid vars string",
                        lineNum);
                }

                int index = Integer.parseInt(matcher.group(1));
                varItems.set(
                    index,
                    new VarItem(matcher.group(2), matcher.group(4)));
            } else if (line.contains("vars[0]")) {
                System.out.println(line);
            }

            matcher = ITERATIONS_P.matcher(line);
            if (matcher.matches()) {
                if (iterations != -1) {
                    throw new MatrixnetModelParseException(
                        "Two times iterations num parsed",
                        lineNum);
                }

                iterations = Integer.parseInt(matcher.group(1));
            }

            matcher = SCORE_WEIGHT_P.matcher(line);
            if (matcher.matches()) {
                if (scoreOffset != -1 || scoreMultiplier != -1) {
                    throw new MatrixnetModelParseException(
                        "Two times score equation parsed",
                        lineNum);
                }

                if (matcher.groupCount() != 3) {
                    throw new MatrixnetModelParseException(
                        "Unable parse score result formula",
                        lineNum
                    );
                }

                scoreOffset = Double.parseDouble(matcher.group(1));
                scoreMultiplier = Double.parseDouble(matcher.group(2));
            }

            switch (status) {
                case PARSING_INDICIES:
                    if (line.contains(END_BRACE)) {
                        status = 0;
                    } else {
                        indiciesList.add(parseArrayString(line));
                    }
                    break;
                case PARSING_DATA:
                    if (line.contains(END_BRACE)) {
                        status = 0;
                    } else {
                        dataList.add(parseArrayString(line));
                    }
                    break;
                case 0:
                    break;
                default:
                    throw new MatrixnetModelParseException(
                        "Unknown state",
                        lineNum);
            }
        }

        indicies = parseArray(indiciesList);
        data = parseArray(dataList);
        if (varItems.isEmpty()) {
            throw new MatrixnetModelParseException(
                "Empty vars list",
                -1);
        }
    }
    // CSON: MagicNumber

    private int[] parseArrayString(final String s) {
        String[] split = s.trim().split(",");
        int[] result = new int[split.length];
        int count = 0;
        for (String item: split) {
            if (item.trim().isEmpty()) {
                continue;
            }

            result[count] = Integer.parseInt(item);
            count += 1;
        }

        return Arrays.copyOf(result, count);
    }

    private int[] parseArray(final List<int[]> list) {
        int count = 0;
        for (int[] items: list) {
            count += items.length;
        }

        int[] result = new int[count];
        int offset = 0;
        for (int[] items: list) {
            for (int i = 0; i < items.length; i++) {
                result[offset] = items[i];
                offset++;
            }
        }

        return result;
    }

    @Override
    public double score(final double[] factors) {
        long score = 0;

        int[] vars = new int[varsNum];
        for (int i = 0; i < varItems.size(); i++) {
            VarItem item = varItems.get(i);
            if (item.empty()) {
                vars[i] = 0;
            } else {
                if (factors[item.factorIndex] > item.threshold) {
                    vars[i] = 1;
                } else {
                    vars[i] = 0;
                }
            }
        }

        int indicesIdx = 0;
        int dataIdx = 0;

        for (int z = 0; z < iterations; ++z) {
            int idx = 0;
            for (int j = INDICIES_ITERS; j >= 0; j--) {
                idx = 2 * idx + vars[indicies[indicesIdx + j]];
            }

            score += data[dataIdx + idx];

            indicesIdx += INDICIES_IDX_INC;
            dataIdx += DATA_IDX_INC;
        }

        return scoreOffset + score * scoreMultiplier;
    }

    public static class MatrixnetModelParseException extends IOException {
        private static final long serialVersionUID = 7610279873340183585L;

        public MatrixnetModelParseException(
            final String message,
            final int lineNumber)
        {
            super(message + " on line number " + lineNumber);
        }

        public MatrixnetModelParseException(final Throwable cause) {
            super(cause);
        }
    }

    public static class VarItem {
        private final int factorIndex;
        private final double threshold;
        private final boolean empty;

        public VarItem() {
            this.factorIndex = 0;
            this.threshold = 0;
            this.empty = true;
        }

        public VarItem(final int factorIndex, final float threshold) {
            this.factorIndex = factorIndex;
            this.threshold = threshold;
            this.empty = false;
        }

        public VarItem(final String fIndex, final String threshold) {
            this.factorIndex = Integer.parseInt(fIndex);
            this.threshold = Double.parseDouble(threshold);
            this.empty = false;
        }

        public int factorIndex() {
            return factorIndex;
        }

        public double threshold() {
            return threshold;
        }

        public boolean empty() {
            return empty;
        }
    }

    protected List<VarItem> vars() {
        return varItems;
    }

    protected int[] indicies() {
        return indicies;
    }

    protected int[] data() {
        return data;
    }

    protected int varsNum() {
        return varsNum;
    }

    protected int iterations() {
        return iterations;
    }

    protected double scoreOffset() {
        return scoreOffset;
    }

    protected double scoreMultiplier() {
        return scoreMultiplier;
    }
}
