package ru.yandex.mail.mime;

import java.util.Locale;

import org.apache.james.mime4j.stream.BodyDescriptorBuilder;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.NameValuePair;
import org.apache.james.mime4j.stream.RawBody;
import org.apache.james.mime4j.stream.RawField;
import org.apache.james.mime4j.stream.RawFieldParser;

import ru.yandex.util.string.StringUtils;

public class OverwritingBodyDescriptorBuilder
    implements BodyDescriptorBuilder
{
    private final String parentMimeType;
    private String mimeType;
    private String mediaType;
    private String subType;
    private String boundary;
    private String charset;
    private String transferEncoding;
    private long contentLength;

    public OverwritingBodyDescriptorBuilder() {
        this(null);
    }

    private OverwritingBodyDescriptorBuilder(final String parentMimeType) {
        this.parentMimeType = parentMimeType;
        reset();
    }

    @Override
    public void reset() {
        mimeType = null;
        mediaType = null;
        subType = null;
        boundary = null;
        charset = null;
        transferEncoding = null;
        contentLength = -1L;
    }

    @Override
    public OverwritingBodyDescriptorBuilder newChild() {
        return new OverwritingBodyDescriptorBuilder(mimeType);
    }

    @Override
    public BasicBodyDescriptor build() {
        String mimeType = this.mimeType;
        String mediaType = this.mediaType;
        String subType = this.subType;
        if (mimeType == null) {
            if ("multipart/digest".equals(parentMimeType)) {
                mimeType = "message/rfc822";
                mediaType = "message";
                subType = "rfc822";
            } else {
                mimeType = "text/plain";
                mediaType = "text";
                mediaType = "plain";
            }
        }

        String charset = this.charset;
        if (charset == null && "text".equals(mediaType)) {
            charset = "utf-8";
        }
        String transferEncoding = this.transferEncoding;
        if (transferEncoding == null) {
            transferEncoding = "7bit";
        }
        return new BasicBodyDescriptor(
            mimeType,
            mediaType,
            subType,
            boundary,
            charset,
            transferEncoding,
            contentLength);
    }

    @Override
    public Field addField(final RawField field) {
        String name = field.getName().toLowerCase(Locale.ROOT);

        // XXX: Deviation from standard, rfc requires to account only first
        // occurence of Content-* header, but Yandex.Mail takes the last one
        switch (name) {
            case "content-transfer-encoding":
                String transferEncoding = field.getBody();
                if (transferEncoding != null) {
                    transferEncoding =
                        transferEncoding.trim().toLowerCase(Locale.ROOT);
                    if (!transferEncoding.isEmpty()) {
                        this.transferEncoding = transferEncoding;
                    }
                }
                break;
            case "content-length":
                String contentLength = field.getBody();
                if (contentLength != null) {
                    try {
                        this.contentLength =
                            Long.parseLong(contentLength.trim());
                    } catch (RuntimeException e) {
                        // Ignore
                    }
                }
                break;
            case "content-type":
                parseContentType(field);
                break;
            default:
                break;
        }
        return null;
    }

    private void parseContentType(final RawField field) {
        RawBody body = RawFieldParser.DEFAULT.parseRawBody(field);
        String boundary = null;
        String charset = null;
        for (NameValuePair pair: body.getParams()) {
            String name = pair.getName();
            if (name.equalsIgnoreCase("charset")) {
                String value = pair.getValue();
                if (value != null) {
                    charset = value.trim();
                }
            } else if (name.equalsIgnoreCase("boundary")) {
                boundary = pair.getValue();
            }
        }

        String mimeType = body.getValue();
        if (mimeType != null) {
            mimeType = mimeType.toLowerCase(Locale.ROOT);
        }
        String mediaType = null;
        String subType = null;

        if (mimeType != null) {
            int idx = mimeType.indexOf('/');
            boolean valid = false;
            if (idx != -1) {
                mediaType = mimeType.substring(0, idx).trim();
                subType = mimeType.substring(idx + 1).trim();
                if (!mediaType.isEmpty() && !subType.isEmpty()) {
                    mimeType = StringUtils.concat(mediaType, '/', subType);
                    valid = true;
                }
            }
            if (!valid) {
                mimeType = null;
                mediaType = null;
                subType = null;
            }
        }

        boolean multipart = "multipart".equals(mediaType);
        if (!multipart || (multipart && boundary != null)) {
            this.mimeType = mimeType;
            this.mediaType = mediaType;
            this.subType = subType;
            this.boundary = boundary;
        }

        if (charset != null && !charset.isEmpty()) {
            this.charset = charset;
        }
    }
}

