package ru.yandex.mail.mime;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.FieldBuilder;
import org.apache.james.mime4j.stream.RawField;
import org.apache.james.mime4j.util.ByteArrayBuffer;
import org.mozilla.universalchardet.CharsetListener;
import org.mozilla.universalchardet.UniversalDetector;

import ru.yandex.charset.Decoder;

public class Utf8FieldBuilder implements CharsetListener, FieldBuilder {
    private static final int INITIAL_SIZE = 256;
    private static final char[] EMPTY_BUF = new char[0];

    private final ByteArrayBuffer buf = new ByteArrayBuffer(INITIAL_SIZE);
    private char[] latin1 = EMPTY_BUF;
    private Charset charset = null;
    private Charset lastUsedCharset = null;
    private Decoder decoder = null;

    @Override
    public void report(final String charsetName) {
        try {
            charset = Charset.forName(charsetName);
        } catch (RuntimeException e) {
        }
    }

    @Override
    public void reset() {
        buf.clear();
    }

    @Override
    public ByteArrayBuffer getRaw() {
        return buf;
    }

    @Override
    public void append(final ByteArrayBuffer line) {
        if (line != null) {
            buf.append(line.buffer(), 0, line.length());
        }
    }

    @Override
    public RawField build() throws MimeException {
        byte[] buf = this.buf.buffer();
        int len = this.buf.length();
        if (len > 0) {
            if (buf[len - 1] == '\n') {
                --len;
            }
        }
        if (len > 0) {
            if (buf[len - 1] == '\r') {
                --len;
            }
        }

        int fieldNameLength = fieldNameLength(buf, len);
        String name = latin1(buf, 0, fieldNameLength);

        int bodyStart = fieldNameLength + 1;
        if (bodyStart + 1 < len) {
            switch (buf[bodyStart]) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    ++bodyStart;
                    break;
                default:
                    break;
            }
        }

        // We found start of body. Let's unfold it in place
        int unfoldIndex = bodyStart;
        for (; unfoldIndex < len; ++unfoldIndex) {
            byte b = buf[unfoldIndex];
            if (b == '\r' || b == '\n') {
                break;
            }
        }
        if (unfoldIndex < len) {
            int shift = 1;
            for (int i = unfoldIndex + 1; i < len; ++i) {
                byte b = buf[i];
                if (b == '\r' || b == '\n') {
                    ++shift;
                } else {
                    buf[i - shift] = buf[i];
                }
            }
            len -= shift;
        }

        boolean ascii = true;
        for (int i = bodyStart; i < len; ++i) {
            if (buf[i] < 0) {
                ascii = false;
                break;
            }
        }

        int bodyLength = len - bodyStart;
        String body;
        if (ascii) {
            body = latin1(buf, bodyStart, bodyLength);
        } else {
            charset = null;
            UniversalDetector detector = new UniversalDetector(this);
            detector.handleData(buf, bodyStart, bodyLength);
            detector.dataEnd();
            if (charset == null) {
                if (!StandardCharsets.UTF_8.equals(lastUsedCharset)) {
                    lastUsedCharset = StandardCharsets.UTF_8;
                    decoder = new Decoder(StandardCharsets.UTF_8);
                }
            } else {
                if (!charset.equals(lastUsedCharset)) {
                    lastUsedCharset = charset;
                    decoder = new Decoder(charset);
                }
            }
            try {
                decoder.process(buf, bodyStart, bodyLength);
                body = decoder.toString();
            } catch (IOException e) {
                body = latin1(buf, bodyStart, bodyLength);
            }
        }

        return new RawField(name, body);
    }

    private String latin1(final byte[] buf, final int off, final int len) {
        if (latin1.length < len) {
            latin1 = new char[Math.max(len, latin1.length << 1)];
        }
        for (int i = 0; i < len; ++i) {
            latin1[i] = (char) (buf[i + off] & 0xff);
        }
        return new String(latin1, 0, len);
    }

    private int fieldNameLength(final byte[] buf, final int len)
        throws MimeException
    {
        for (int i = 0; i < len; ++i) {
            byte b = buf[i];
            if (b == ':') {
                return i;
            } else if (b <= ' ' || b == 0x7f) {
                throw new MimeException(
                    "Bad character 0x" + Integer.toHexString(b & 0xff)
                    + " at pos " + i
                    + " while parsing <" + latin1(buf, 0, len) + '>');
            }
        }
        throw new MimeException(
            "Field name delimiter not found in <" + latin1(buf, 0, len) + '>');
    }
}

