package ru.yandex.parser.rfc2047;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.charset.Decoder;
import ru.yandex.function.StringVoidProcessor;
import ru.yandex.io.DecodableByteArrayOutputStream;

public interface Rfc2047Parser {
    // TODO: use match table from EmailParser
    Pattern RFC2047_ENCODED_WORD = Pattern.compile(
        "=\\?([a-zA-Z0-9+.:_-]+?)\\?([bBqQ])\\?(.*?)\\?=",
        Pattern.DOTALL);

    static boolean isWhitespace(
        final String str,
        final int off,
        final int end)
    {
        for (int i = off; i < end; ++i) {
            switch (str.charAt(i)) {
                case ' ':
                case '\t':
                case '\n':
                case '\r':
                    break;
                default:
                    return false;
            }
        }
        return true;
    }

    static String decode(
        final String body,
        final Rfc2047DecodersProvider decodersProvider)
    {
        int idx = body.indexOf("=?");
        String result;
        if (idx == -1) {
            result = body;
        } else {
            StringBuilder sb = decodersProvider.stringBuilder(body.length());
            sb.append(body, 0, idx);
            DecodableByteArrayOutputStream byteArray =
                decodersProvider.byteArray(body.length() >> 1);
            int tailIndex = 0;
            boolean lastMatchValid = false;
            Charset prevCharset = null;
            String encoded = body.substring(idx);
            Matcher matcher = RFC2047_ENCODED_WORD.matcher(encoded);
            while (matcher.find()) {
                int start = matcher.start();
                String charsetName = matcher.group(1);
                Charset charset;
                try {
                    charset = Charset.forName(charsetName);
                } catch (RuntimeException e) {
                    charset = null;
                }
                boolean skippableSeparator =
                    start == tailIndex
                    || (lastMatchValid
                        && isWhitespace(encoded, tailIndex, start));
                if (charset == null
                    || !skippableSeparator
                    || !charset.equals(prevCharset))
                {
                    if (!byteArray.isEmpty()) {
                        Decoder decoder =
                            decodersProvider.decoderFor(prevCharset);
                        try {
                            byteArray.processWith(decoder);
                        } catch (IOException e) {
                            // Impossible
                        }
                        decoder.toStringBuilder(sb);
                        byteArray.reset();
                    }
                    prevCharset = null;
                }
                if (charset == null) {
                    lastMatchValid = false;
                    continue;
                }
                prevCharset = charset;
                char type = encoded.charAt(matcher.start(2));
                StringVoidProcessor<byte[], IOException> decoder;
                if (type == 'b' || type == 'B') {
                    decoder = decodersProvider.base64Decoder();
                } else {
                    decoder = decodersProvider.quotedPrintableDecoder();
                }
                try {
                    int begin = matcher.start(2 + 1);
                    int end = matcher.end(2 + 1);
                    decoder.process(encoded, begin, end - begin);
                } catch (IOException e) {
                    lastMatchValid = false;
                    continue;
                }
                decoder.processWith(byteArray);
                if (!skippableSeparator) {
                    sb.append(encoded, tailIndex, start);
                }
                tailIndex = matcher.end();
                lastMatchValid = true;
            }
            if (tailIndex == 0) {
                result = body;
            } else {
                if (!byteArray.isEmpty()) {
                    Decoder decoder = decodersProvider.decoderFor(prevCharset);
                    try {
                        byteArray.processWith(decoder);
                    } catch (IOException e) {
                        // impossible
                    }
                    decoder.toStringBuilder(sb);
                }
                sb.append(encoded, tailIndex, encoded.length());
                result = sb.toString();
            }
        }
        return result;
    }

}

