package ru.yandex.parser.string;

import ru.yandex.function.GenericFunction;

public enum DurationParser
    implements GenericFunction<String, Double, RuntimeException>
{
    INSTANCE;

    public static final GenericFunction<String, Integer, Exception> INTEGER =
        DurationParser.INSTANCE
            .andThen(NonNegativeValidator.instance())
            .andThen(DoubleToIntegerValidator.INSTANCE);
    public static final GenericFunction<String, Long, Exception> NEGATIVELY_LONG =
        DurationParser.INSTANCE
            .andThen(DoubleToLongValidator.INSTANCE);
    public static final GenericFunction<String, Long, Exception> LONG =
        DurationParser.INSTANCE
            .andThen(NonNegativeValidator.instance())
            .andThen(DoubleToLongValidator.INSTANCE);
    public static final GenericFunction<String, Long, Exception>
        POSITIVE_LONG = DurationParser.INSTANCE
            .andThen(PositiveValidator.instance())
            .andThen(DoubleToLongValidator.INSTANCE);

    private static final double MICROS_MULTIPLIER = 0.001d;
    private static final double NANOS_MULTIPLIER = 0.000001d;
    private static final double MILLIS_PER_SECOND = 1000d;

    private static final int NONE = 0;
    private static final int SECONDS = 1;
    private static final int MINUTES = 2;
    private static final int HOURS = 3;
    private static final int DAYS = 4;
    private static final int WEEKS = 5;

    private static final double[] MULTIPLIERS = new double[] {
        1d,
        0d,
        60000d,
        3600000d,
        86400000d,
        604800000d
    };

    @Override
    public Double apply(final String value) {
        int end = value.length() - 1;
        char last = value.charAt(end);
        int type;
        switch (last) {
            case 's':
                type = SECONDS;
                break;
            case 'm':
                type = MINUTES;
                break;
            case 'h':
                type = HOURS;
                break;
            case 'd':
                type = DAYS;
                break;
            case 'w':
                type = WEEKS;
                break;
            default:
                type = NONE;
                break;
        }
        double result;
        if (type == NONE) {
            result = Double.parseDouble(value);
        } else {
            double multiplier;
            if (type == SECONDS) {
                switch (value.charAt(end - 1)) {
                    case 'm':
                        --end;
                        multiplier = 1d;
                        break;
                    case 'n':
                        --end;
                        multiplier = NANOS_MULTIPLIER;
                        break;
                    case 'µ':
                        --end;
                        multiplier = MICROS_MULTIPLIER;
                        break;
                    default:
                        multiplier = MILLIS_PER_SECOND;
                        break;
                }
            } else {
                multiplier = MULTIPLIERS[type];
            }
            if (value.charAt(end - 1) == ' ') {
                --end;
            }
            result = Double.parseDouble(value.substring(0, end)) * multiplier;
        }
        return Math.ceil(result);
    }
}

