package ru.yandex.parser.string;

import java.util.Map;
import java.util.function.Supplier;

import ru.yandex.function.CharArrayProcessor;
import ru.yandex.function.GenericFunction;

public class MapParser<K, V>
    implements GenericFunction<String, Map<K, V>, Exception>, CharArrayProcessor<Map<K, V>, Exception>
{
    private final Supplier<Map<K, V>> mapSupplier;
    private final GenericFunction<String, K, Exception> keyParser;
    private final GenericFunction<String, V, Exception> valueParser;
    private final char recordsDelimiter;
    private final char keyValueDelimiter;

    public MapParser(
        final Supplier<Map<K, V>> mapSupplier,
        final GenericFunction<String, K, Exception> keyParser,
        final GenericFunction<String, V, Exception> valueParser)
    {
        this(mapSupplier, keyParser, valueParser, '\n', '\t');
    }

    public MapParser(
        final Supplier<Map<K, V>> mapSupplier,
        final GenericFunction<String, K, Exception> keyParser,
        final GenericFunction<String, V, Exception> valueParser,
        final char recordsDelimiter,
        final char keyValueDelimiter)
    {
        this.mapSupplier = mapSupplier;
        this.keyParser = keyParser;
        this.valueParser = valueParser;
        this.recordsDelimiter = recordsDelimiter;
        this.keyValueDelimiter = keyValueDelimiter;
    }

    @Override
    public Map<K, V> apply(final String value) throws Exception {
        return process(value.toCharArray());
    }

    @Override
    public Map<K, V> process(
        final char[] buf,
        final int off,
        final int len)
        throws Exception
    {
        Map<K, V> map = mapSupplier.get();

        StringBuilder keySb = new StringBuilder(len);
        StringBuilder valueSb = new StringBuilder(len);
        StringBuilder sb = keySb;
        int state = 0;
        boolean escaped = false;
        for (int i = 0; i < len; ++i) {
            char c = buf[i + off];
            if (escaped) {
                if (c == '\\' || c == recordsDelimiter || c == keyValueDelimiter) {
                    sb.append(c);
                } else {
                    sb.append('\\');
                    sb.append(c);
                }
                escaped = false;
            } else if (c == '\\') {
                escaped = true;
            } else if (c == keyValueDelimiter) {
                if (state != 0) {
                    throw new Exception("Invalid parser state, expecting 0 but got " + state);
                }

                state = 1;
                sb = valueSb;
            } else if (c == recordsDelimiter) {
                if (state != 1) {
                    throw new Exception("Invalid parser state, expecting 1 but got " + state);
                }
                map.put(
                    keyParser.apply(keySb.toString()),
                    valueParser.apply(valueSb.toString()));
                keySb.setLength(0);
                valueSb.setLength(0);
                sb = keySb;
                state = 0;
            } else {
                sb.append(c);
            }
        }
        if (escaped) {
            sb.append('\\');
        }
        if (state != 1 && sb.length() > 0) {
            throw new Exception("Invalid parser state, extra data in end " + sb.toString());
        }

        if (state == 1) {
            map.put(
                keyParser.apply(keySb.toString()),
                valueParser.apply(valueSb.toString()));
            keySb.setLength(0);
            valueSb.setLength(0);
        }

        return map;
    }
}
