package ru.yandex.parser.string;

import java.net.URI;
import java.net.URISyntaxException;

import ru.yandex.function.GenericFunction;
import ru.yandex.net.uri.fast.FastUri;
import ru.yandex.net.uri.fast.FastUriParser;

public enum URIParser
    implements GenericFunction<String, URI, URISyntaxException>
{
    INSTANCE {
        @Override
        public URI apply(final String value) throws URISyntaxException {
            return parse(value);
        }
    },
    FRAGMENT_STRIPPING {
        @Override
        public URI apply(final String value) throws URISyntaxException {
            return stripFragment(parse(value));
        }
    };

    public static URI stripFragment(final URI uri) throws URISyntaxException {
        if (uri.getRawFragment() == null) {
            return uri;
        } else {
            return createURI(
                uri.getScheme(),
                uri.getRawUserInfo(),
                uri.getHost(),
                uri.getPort(),
                uri.getRawPath(),
                uri.getRawQuery(),
                null,
                uri.getRawSchemeSpecificPart(),
                uri.getRawAuthority());
        }
    }

    public static String createURIString(
        final String scheme,
        final String userInfo,
        final String host,
        final int port,
        final String path,
        final String query,
        final String fragment,
        final String schemeSpecificPart,
        final String authority)
    {
        return new FastUri(
            scheme,
            userInfo,
            host,
            port,
            path,
            query,
            fragment,
            schemeSpecificPart,
            authority)
            .toString();
    }

    public static URI createURI(
        final String scheme,
        final String userInfo,
        final String host,
        final int port,
        final String path,
        final String query,
        final String fragment,
        final String schemeSpecificPart,
        final String authority)
        throws URISyntaxException
    {
        String uri = createURIString(
            scheme,
            userInfo,
            host,
            port,
            path,
            query,
            fragment,
            schemeSpecificPart,
            authority);
        return new URI(uri).parseServerAuthority();
    }

    public static URI parse(final String uri) throws URISyntaxException {
        FastUri fastUri = new FastUriParser(uri).parse();
        return createURI(
            fastUri.scheme(),
            fastUri.userInfo(),
            fastUri.host(),
            fastUri.port(),
            fastUri.path(),
            fastUri.query(),
            fastUri.fragment(),
            fastUri.schemeSpecificPart(),
            fastUri.authority());
    }
}

