package ru.yandex.parser.string;

import java.io.File;
import java.math.BigInteger;
import java.net.URI;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.Iterator;
import java.util.Locale;
import java.util.logging.Level;

import ru.yandex.function.GenericFunction;

public interface ValuesStorage<E extends Exception> {
    E parameterNotSetException(String name);

    E parseFailedException(String name, String value, Throwable cause);

    String getOrNull(String name);

    String getLastOrNull(String name);

    Iterator<String> getAllOrNull(String name);

    default <T> T parse(
        final String name,
        final String value,
        final GenericFunction<String, ? extends T, ? extends Exception> parser)
        throws E
    {
        try {
            return parser.apply(value);
        } catch (Exception e) {
            throw parseFailedException(name, value, e);
        }
    }

    // CSOFF: ParameterNumber
    default <T, C extends Collection<? super T>> C parse(
        final String name,
        final Iterator<String> values,
        final GenericFunction<String, ? extends T, ? extends Exception> parser,
        final C collection)
        throws E
    {
        while (values.hasNext()) {
            String value = values.next();
            if (value == null) {
                break;
            }
            T t;
            try {
                t = parser.apply(value);
            } catch (Exception e) {
                throw parseFailedException(name, value, e);
            }
            collection.add(t);
        }
        return collection;
    }

    default <T, C extends Collection<T>> C parse(
        final String name,
        final Iterator<String> values,
        final GenericFunction<String, ? extends C, ? extends Exception> parser)
        throws E
    {
        String value = values.next();
        C collection;
        try {
            collection = parser.apply(value);
        } catch (Exception e) {
            throw parseFailedException(name, value, e);
        }
        while (values.hasNext()) {
            value = values.next();
            try {
                collection.addAll(parser.apply(value));
            } catch (Exception e) {
                throw parseFailedException(name, value, e);
            }
        }
        return collection;
    }
    // CSON: ParameterNumber

    default <T> T get(
        final String name,
        final GenericFunction<String, ? extends T, ? extends Exception> parser)
        throws E
    {
        String value = getOrNull(name);
        if (value == null) {
            throw parameterNotSetException(name);
        } else {
            return parse(name, value, parser);
        }
    }

    default <T> T get(
        final String name,
        final T defaultValue,
        final GenericFunction<String, ? extends T, ? extends Exception> parser)
        throws E
    {
        String value = getOrNull(name);
        if (value == null) {
            return defaultValue;
        } else {
            return parse(name, value, parser);
        }
    }

    default <T> T getLast(
        final String name,
        final GenericFunction<String, ? extends T, ? extends Exception> parser)
        throws E
    {
        String value = getLastOrNull(name);
        if (value == null) {
            throw parameterNotSetException(name);
        } else {
            return parse(name, value, parser);
        }
    }

    default <T> T getLast(
        final String name,
        final T defaultValue,
        final GenericFunction<String, ? extends T, ? extends Exception> parser)
        throws E
    {
        String value = getLastOrNull(name);
        if (value == null) {
            return defaultValue;
        } else {
            return parse(name, value, parser);
        }
    }

    default <T, C extends Collection<? super T>> C getAll(
        final String name,
        final GenericFunction<String, ? extends T, ? extends Exception> parser,
        final C collection)
        throws E
    {
        Iterator<String> iter = getAllOrNull(name);
        if (iter == null) {
            throw parameterNotSetException(name);
        } else {
            return parse(name, iter, parser, collection);
        }
    }

    // CSOFF: ParameterNumber
    default <T, C extends Collection<? super T>> C getAll(
        final String name,
        final C defaultValue,
        final GenericFunction<String, ? extends T, ? extends Exception> parser,
        final C collection)
        throws E
    {
        Iterator<String> iter = getAllOrNull(name);
        if (iter == null) {
            return defaultValue;
        } else {
            return parse(name, iter, parser, collection);
        }
    }
    // CSON: ParameterNumber

    default <T, C extends Collection<T>> C getAll(
        final String name,
        final GenericFunction<String, ? extends C, ? extends Exception> parser)
        throws E
    {
        Iterator<String> iter = getAllOrNull(name);
        if (iter == null) {
            throw parameterNotSetException(name);
        } else {
            return parse(name, iter, parser);
        }
    }

    default <T, C extends Collection<T>> C getAll(
        final String name,
        final C defaultValue,
        final GenericFunction<String, ? extends C, ? extends Exception> parser)
        throws E
    {
        Iterator<String> iter = getAllOrNull(name);
        if (iter == null) {
            return defaultValue;
        } else {
            return parse(name, iter, parser);
        }
    }

    default String getString(final String name) throws E {
        return get(name, x -> x);
    }

    default String getString(final String name, final String defaultValue) {
        String value = getOrNull(name);
        if (value == null) {
            return defaultValue;
        } else {
            return value;
        }
    }

    default int getInt(final String name) throws E {
        return get(name, IntegerParser.INSTANCE);
    }

    default Integer getInt(final String name, final Integer defaultValue)
        throws E
    {
        return get(name, defaultValue, IntegerParser.INSTANCE);
    }

    default long getLong(final String name) throws E {
        return get(name, LongParser.INSTANCE);
    }

    default Long getLong(final String name, final Long defaultValue) throws E {
        return get(name, defaultValue, LongParser.INSTANCE);
    }

    default double getDouble(final String name) throws E {
        return get(name, DoubleParser.INSTANCE);
    }

    default Double getDouble(final String name, final Double defaultValue)
        throws E
    {
        return get(name, defaultValue, DoubleParser.INSTANCE);
    }

    default BigInteger getBigInteger(final String name) throws E {
        return get(name, BigIntegerParser.INSTANCE);
    }

    default BigInteger getBigInteger(
        final String name,
        final BigInteger defaultValue)
        throws E
    {
        return get(name, defaultValue, BigIntegerParser.INSTANCE);
    }

    default boolean getBoolean(final String name) throws E {
        return get(name, BooleanParser.INSTANCE);
    }

    default Boolean getBoolean(final String name, final Boolean defaultValue)
        throws E
    {
        return get(name, defaultValue, BooleanParser.INSTANCE);
    }

    default <T extends Enum<T>> T getEnum(final Class<T> t, final String name)
        throws E
    {
        return get(name, new EnumParser<>(t));
    }

    default <T extends Enum<T>> T getEnum(
        final Class<T> t,
        final String name,
        final T defaultValue)
        throws E
    {
        return get(name, defaultValue, new EnumParser<>(t));
    }

    default File getInputFile(final String name) throws E {
        return get(name, FileParser.INPUT_FILE);
    }

    default File getInputFile(final String name, final File defaultValue)
        throws E
    {
        return get(name, defaultValue, FileParser.INPUT_FILE);
    }

    default File getOutputFile(final String name) throws E {
        return get(name, FileParser.OUTPUT_FILE);
    }

    default File getOutputFile(final String name, final File defaultValue)
        throws E
    {
        return get(name, defaultValue, FileParser.OUTPUT_FILE);
    }

    default File getDir(final String name) throws E {
        return get(name, FileParser.DIRECTORY);
    }

    default File getDir(final String name, final File defaultValue) throws E {
        return get(name, defaultValue, FileParser.DIRECTORY);
    }

    default File getExistingDir(final String name) throws E {
        return get(name, FileParser.EXISTING_DIRECTORY);
    }

    default File getExistingDir(final String name, final File defaultValue)
        throws E
    {
        return get(name, defaultValue, FileParser.EXISTING_DIRECTORY);
    }

    default Charset getCharset(final String name) throws E {
        return get(name, CharsetParser.INSTANCE);
    }

    default Charset getCharset(final String name, final Charset defaultValue)
        throws E
    {
        return get(name, defaultValue, CharsetParser.INSTANCE);
    }

    default Locale getLocale(final String name) throws E {
        return get(name, LocaleParser.INSTANCE);
    }

    default Locale getLocale(final String name, final Locale defaultValue)
        throws E
    {
        return get(name, defaultValue, LocaleParser.INSTANCE);
    }

    default Level getLogLevel(final String name) throws E {
        return get(name, LogLevelParser.INSTANCE);
    }

    default Level getLogLevel(final String name, final Level defaultValue)
        throws E
    {
        return get(name, defaultValue, LogLevelParser.INSTANCE);
    }

    default URI getURI(final String name) throws E {
        return get(name, URIParser.INSTANCE);
    }

    default URI getURI(final String name, final URI defaultValue) throws E {
        return get(name, defaultValue, URIParser.INSTANCE);
    }

    default int getIntegerDuration(final String name) throws E {
        return get(name, DurationParser.INTEGER);
    }

    default Integer getIntegerDuration(
        final String name,
        final Integer defaultValue)
        throws E
    {
        return get(name, defaultValue, DurationParser.INTEGER);
    }

    default long getLongDuration(final String name) throws E {
        return get(name, DurationParser.LONG);
    }

    default Long getLongDuration(final String name, final Long defaultValue)
        throws E
    {
        return get(name, defaultValue, DurationParser.LONG);
    }

    default Long getNegativelyLongDuration(final String name, final Long defaultValue)
            throws E
    {
        return get(name, defaultValue, DurationParser.NEGATIVELY_LONG);
    }
}

