package ru.yandex.util.string;

import com.ibm.icu.text.Normalizer2;

import ru.yandex.function.CharArrayProcessable;
import ru.yandex.function.CharArrayVoidProcessor;
import ru.yandex.function.Processor;
import ru.yandex.function.VoidProcessor;

public class NormalizingProcessor
    extends CharArrayProcessable
    implements CharArrayVoidProcessor<RuntimeException>
{
    public static final int NORMALIZATION_BLOCK_SIZE = 1024;
    public static final int MAX_LOOKAHEAD = 4;

    private static final Normalizer2 NORMALIZER = Normalizer2.getNFCInstance();

    private final boolean allowDataSharing;
    private final ReusableCharSequence seq = new ReusableCharSequence();
    private StringBuilder sb = null;
    // Indicates that buf can be used by someone else
    private boolean shared = false;
    private boolean useSB = false;

    public NormalizingProcessor() {
        this(false);
    }

    // Data copying can be avoided if no data is stored and all .process(...)
    // invocations are followed immediately by .processWith(...) or .toString()
    public NormalizingProcessor(final boolean allowDataSharing) {
        this.allowDataSharing = allowDataSharing;
    }

    public static boolean good(final char c) {
        return NORMALIZER.hasBoundaryBefore(c);
    }

    @Override
    public void process(final char[] buf, final int off, final int len) {
        seq.buf(buf, off, len);
        int badPos = NORMALIZER.spanQuickCheckYes(seq);
        if (badPos == len) {
            if (allowDataSharing) {
                this.buf = buf;
                this.off = off;
                this.len = len;
                shared = true;
            } else {
                ensureBufCapacity(len);
                copyFrom(buf, off, len);
            }
            useSB = false;
        } else {
            useSB = true;
            shared = false;
            if (sb == null) {
                sb = new StringBuilder(len + MAX_LOOKAHEAD);
            } else {
                sb.setLength(0);
            }
            NORMALIZER.normalize(seq, sb);
            this.len = sb.length();
            this.off = 0;
        }
    }

    @Override
    public <E extends Exception> void processWith(
        final VoidProcessor<? super char[], E> processor)
        throws E
    {
        if (useSB) {
            if (shared || this.buf.length < this.len) {
                this.buf = new char[this.len];
            }
            sb.getChars(this.off, this.off + this.len, this.buf, 0);
        }
        super.processWith(processor);
    }

    @Override
    public <R, E extends Exception> R processWith(
        final Processor<? super char[], R, E> processor)
        throws E
    {
        if (useSB) {
            if (shared || this.buf.length < this.len) {
                this.buf = new char[this.len];
            }
            sb.getChars(this.off, this.off + this.len, this.buf, 0);
        }
        return super.processWith(processor);
    }

    @Override
    public int transferTo(final char[] buf, final int off, final int len) {
        int transferSize = Math.min(len, this.len);
        if (useSB) {
            sb.getChars(this.off, this.off + transferSize, buf, off);
        } else {
            System.arraycopy(this.buf, this.off, buf, off, transferSize);
        }
        this.off += transferSize;
        this.len -= transferSize;
        return transferSize;
    }

    // Made protected because this class doesn't have ensureBufCapacity method
    @Override
    protected void copyFrom(final char[] buf, final int off, final int len) {
        System.arraycopy(buf, off, this.buf, 0, len);
        this.off = 0;
        this.len = len;
    }

    @Override
    public String toString() {
        if (useSB) {
            return new String(sb);
        } else {
            return new String(buf, off, len);
        }
    }

    private static class ReusableCharSequence implements CharSequence {
        private char[] buf;
        private int off;
        private int len;

        ReusableCharSequence() {
            buf = null;
        }

        public void buf(final char[] buf, final int off, final int len) {
            this.buf = buf;
            this.off = off;
            this.len = len;
        }

        @Override
        public char charAt(final int index) {
            return buf[off + index];
        }

        @Override
        public int length() {
            return len;
        }

        @Override
        public CharSequence subSequence(final int start, final int end) {
            throw new UnsupportedOperationException();
        }

        @Override
        public String toString() {
            throw new UnsupportedOperationException();
        }
    }
}
