package ru.yandex.util.string;

import java.io.IOException;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.function.BiConsumer;
import java.util.function.IntPredicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.function.StringBuilderable;

public interface StringUtils {
    int BUFFER_SIZE = 2048;
    int STRING_HASH_CODE_MULTIPLIER = 31;

    static String from(final Reader reader) throws IOException {
        char[] buf = new char[BUFFER_SIZE];
        StringBuilder sb = new StringBuilder();
        while (true) {
            int read = reader.read(buf);
            if (read == -1) {
                return new String(sb);
            }
            sb.append(buf, 0, read);
        }
    }

    static byte[] getUtf8Bytes(final String value) {
        try {
            return value.getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            // Impossible case
            return value.getBytes(StandardCharsets.UTF_8);
        }
    }

    static String nullifyEmpty(final String value) {
        if (value == null || value.isEmpty()) {
            return null;
        } else {
            return value;
        }
    }

    static String toLowerCaseOrNull(final String value) {
        return toLowerCaseOrNull(value, Locale.ROOT);
    }

    static String toLowerCaseOrNull(final String value, final Locale locale) {
        if (value == null) {
            return null;
        } else {
            return value.toLowerCase(locale);
        }
    }

    static String intern(final String value) {
        if (value == null) {
            return null;
        } else {
            return value.intern();
        }
    }

    // If both prefix and suffix are non-null, concatenates them
    // Otherwise returns non-null part
    static String concatOrSet(
        final String prefix,
        final char delimiter,
        final String suffix)
    {
        if (prefix == null) {
            return suffix;
        } else if (suffix == null) {
            return prefix;
        } else {
            return concat(prefix, delimiter, suffix);
        }
    }

    static boolean equalsIgnoreCase(final char c1, final char c2) {
        return c1 == c2
            || Character.toUpperCase(c1) == Character.toUpperCase(c2)
            || Character.toLowerCase(c1) == Character.toLowerCase(c2);
    }

    static int indexOfIgnoreCase(final String haystack, final String needle) {
        return indexOfIgnoreCase(haystack, needle, 0);
    }

    static int indexOfIgnoreCase(
        final String haystack,
        final String needle,
        final int fromIndex)
    {
        int needleLength = needle.length();
        int end = haystack.length() - needleLength;
        char first = needle.charAt(0);
        char firstUpper = Character.toUpperCase(first);
        char firstLower = Character.toLowerCase(first);
        for (int i = fromIndex; i <= end; ++i) {
            char c = haystack.charAt(i);
            if (c == first
                || Character.toUpperCase(c) == firstUpper
                || Character.toLowerCase(c) == firstLower)
            {
                boolean found = true;
                for (int j = 1; j < needleLength && found; ++j) {
                    found = equalsIgnoreCase(
                        haystack.charAt(i + j),
                        needle.charAt(j));
                }
                if (found) {
                    return i;
                }
            }
        }
        return -1;
    }

    static boolean startsWith(final String haystack, final char needle) {
        return !haystack.isEmpty() && haystack.charAt(0) == needle;
    }

    static boolean endsWith(final String haystack, final char needle) {
        int len = haystack.length();
        return len > 0 && haystack.charAt(len - 1) == needle;
    }

    static boolean startsWithIgnoreCase(
        final String haystack,
        final String needle)
    {
        return startsWithIgnoreCase(haystack, needle, 0);
    }

    static boolean startsWithIgnoreCase(
        final String haystack,
        final String needle,
        final int fromIndex)
    {
        int needleLength = needle.length();
        if (fromIndex > haystack.length() - needleLength) {
            return false;
        }
        for (int i = 0; i < needleLength; ++i) {
            if (!equalsIgnoreCase(
                    haystack.charAt(i + fromIndex),
                    needle.charAt(i)))
            {
                return false;
            }
        }
        return true;
    }

    static String removePrefix(final String str, final char prefix) {
        if (str.length() >= 1 && str.charAt(0) == prefix) {
            return str.substring(1);
        } else {
            return str;
        }
    }

    static String removePrefix(final String str, final String prefix) {
        if (str.startsWith(prefix)) {
            return str.substring(prefix.length());
        } else {
            return str;
        }
    }

    static String removeSuffix(final String str, final char suffix) {
        int len = str.length();
        if (len > 0 && str.charAt(len - 1) == suffix) {
            return str.substring(0, len - 1);
        } else {
            return str;
        }
    }

    static String firstLine(final String str) {
        int idx = str.indexOf('\n');
        if (idx == -1) {
            return str;
        } else {
            return str.substring(0, idx);
        }
    }

    static int strlen(final String str) {
        if (str == null) {
            return 2 + 2;
        } else {
            return str.length();
        }
    }

    static String concat(final String str1, final String str2) {
        String result;
        int len1 = strlen(str1);
        if (len1 == 0) {
            result = str2;
        } else {
            int len2 = strlen(str2);
            if (len2 == 0) {
                result = str1;
            } else {
                StringBuilder sb = new StringBuilder(len1 + len2);
                sb.append(str1);
                sb.append(str2);
                result = new String(sb);
            }
        }
        return result;
    }

    static String concat(
        final String str1,
        final String str2,
        final String str3)
    {
        StringBuilder sb =
            new StringBuilder(strlen(str1) + strlen(str2) + strlen(str3));
        sb.append(str1);
        sb.append(str2);
        sb.append(str3);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final char delimiter,
        final String str2)
    {
        StringBuilder sb = new StringBuilder(strlen(str1) + strlen(str2) + 1);
        sb.append(str1);
        sb.append(delimiter);
        sb.append(str2);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final String str2,
        final char suffix)
    {
        StringBuilder sb = new StringBuilder(strlen(str1) + strlen(str2) + 1);
        sb.append(str1);
        sb.append(str2);
        sb.append(suffix);
        return new String(sb);
    }

    static String concat(final char prefix, final String str) {
        StringBuilder sb = new StringBuilder(strlen(str) + 1);
        sb.append(prefix);
        sb.append(str);
        return new String(sb);
    }

    static String concat(final String str, final char suffix) {
        StringBuilder sb = new StringBuilder(strlen(str) + 1);
        sb.append(str);
        sb.append(suffix);
        return new String(sb);
    }

    static String concat(
        final char prefix,
        final String str,
        final char suffix)
    {
        StringBuilder sb = new StringBuilder(strlen(str) + 2);
        sb.append(prefix);
        sb.append(str);
        sb.append(suffix);
        return new String(sb);
    }

    static String concat(
        final char prefix,
        final String str1,
        final String str2,
        final String str3)
    {
        StringBuilder sb =
            new StringBuilder(strlen(str1) + strlen(str2) + strlen(str3) + 1);
        sb.append(prefix);
        sb.append(str1);
        sb.append(str2);
        sb.append(str3);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final String str2,
        final String str3,
        final char suffix)
    {
        StringBuilder sb =
            new StringBuilder(strlen(str1) + strlen(str2) + strlen(str3) + 1);
        sb.append(str1);
        sb.append(str2);
        sb.append(str3);
        sb.append(suffix);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final char delimiter1,
        final String str2,
        final char delimiter2,
        final String str3)
    {
        StringBuilder sb =
            new StringBuilder(strlen(str1) + strlen(str2) + strlen(str3) + 2);
        sb.append(str1);
        sb.append(delimiter1);
        sb.append(str2);
        sb.append(delimiter2);
        sb.append(str3);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final char delimiter1,
        final String str2,
        final char delimiter2)
    {
        StringBuilder sb = new StringBuilder(strlen(str1) + strlen(str2) + 2);
        sb.append(str1);
        sb.append(delimiter1);
        sb.append(str2);
        sb.append(delimiter2);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final char delimiter,
        final String str2,
        final String str3)
    {
        StringBuilder sb =
            new StringBuilder(strlen(str1) + strlen(str2) + strlen(str3) + 1);
        sb.append(str1);
        sb.append(delimiter);
        sb.append(str2);
        sb.append(str3);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final String str2,
        final char delimiter,
        final String str3)
    {
        StringBuilder sb =
            new StringBuilder(strlen(str1) + strlen(str2) + strlen(str3) + 1);
        sb.append(str1);
        sb.append(str2);
        sb.append(delimiter);
        sb.append(str3);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final String str2,
        final String str3,
        final String str4)
    {
        StringBuilder sb = new StringBuilder(
            strlen(str1)
            + strlen(str2)
            + strlen(str3)
            + strlen(str4));
        sb.append(str1);
        sb.append(str2);
        sb.append(str3);
        sb.append(str4);
        return new String(sb);
    }

    static String concat(
        final String str1,
        final String str2,
        final String str3,
        final String str4,
        final char suffix)
    {
        StringBuilder sb = new StringBuilder(
            strlen(str1)
            + strlen(str2)
            + strlen(str3)
            + strlen(str4)
            + 1);
        sb.append(str1);
        sb.append(str2);
        sb.append(str3);
        sb.append(str4);
        sb.append(suffix);
        return new String(sb);
    }

    static String join(
        final Collection<String> strings,
        final char delimiter)
    {
        return new String(join(strings, delimiter, "", 0));
    }

    static String join(
        final Collection<String> strings,
        final String delimiter)
    {
        return new String(join(strings, delimiter, "", 0));
    }

    static String join(
        final List<String> strings,
        final char delimiter)
    {
        return new String(join(strings, delimiter, "", 0));
    }

    static String join(
        final List<String> strings,
        final String delimiter)
    {
        return new String(join(strings, delimiter, "", 0));
    }

    static String join(final String[] values, final char delimiter) {
        return join(values, delimiter, "", "");
    }

    static String join(final String[] values, final String delimiter) {
        return join(values, delimiter, "", "");
    }

    static String join(
        final Collection<String> strings,
        final char delimiter,
        final String prefix,
        final String suffix)
    {
        return new String(
            join(strings, delimiter, prefix, suffix.length()).append(suffix));
    }

    static String join(
        final Collection<String> strings,
        final String delimiter,
        final String prefix,
        final String suffix)
    {
        return new String(
            join(strings, delimiter, prefix, suffix.length()).append(suffix));
    }

    static String join(
        final List<String> strings,
        final char delimiter,
        final String prefix,
        final String suffix)
    {
        return new String(
            join(strings, delimiter, prefix, suffix.length()).append(suffix));
    }

    static String join(
        final List<String> strings,
        final String delimiter,
        final String prefix,
        final String suffix)
    {
        return new String(
            join(strings, delimiter, prefix, suffix.length()).append(suffix));
    }

    static String join(
        final String[] strings,
        final char delimiter,
        final String prefix,
        final String suffix)
    {
        return new String(
            join(strings, delimiter, prefix, suffix.length()).append(suffix));
    }

    static String join(
        final String[] strings,
        final String delimiter,
        final String prefix,
        final String suffix)
    {
        return new String(
            join(strings, delimiter, prefix, suffix.length()).append(suffix));
    }

    static StringBuilder join(
        final Collection<String> strings,
        final char delimiter,
        final String prefix,
        final int overhead)
    {
        StringBuilder sb;
        int size = strings.size();
        switch (size) {
            case 0:
                sb = new StringBuilder(prefix.length() + overhead);
                sb.append(prefix);
                break;
            case 1:
                String str = strings.iterator().next();
                sb = new StringBuilder(
                    strlen(str) + prefix.length() + overhead);
                sb.append(prefix);
                sb.append(str);
                break;
            default:
                int length = prefix.length() + overhead + size - 1;
                Iterator<String> iter = strings.iterator();
                for (int i = 0; i < size; ++i) {
                    length += strlen(iter.next());
                }
                sb = new StringBuilder(length);
                sb.append(prefix);
                iter = strings.iterator();
                sb.append(iter.next());
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    sb.append(iter.next());
                }
                break;
        }
        return sb;
    }

    static StringBuilder join(
        final Collection<String> strings,
        final String delimiter,
        final String prefix,
        final int overhead)
    {
        StringBuilder sb;
        int size = strings.size();
        switch (size) {
            case 0:
                sb = new StringBuilder(prefix.length() + overhead);
                sb.append(prefix);
                break;
            case 1:
                String str = strings.iterator().next();
                sb = new StringBuilder(
                    strlen(str) + prefix.length() + overhead);
                sb.append(prefix);
                sb.append(str);
                break;
            default:
                int length =
                    prefix.length() + overhead
                    + (size - 1) * delimiter.length();
                Iterator<String> iter = strings.iterator();
                for (int i = 0; i < size; ++i) {
                    length += strlen(iter.next());
                }
                sb = new StringBuilder(length);
                sb.append(prefix);
                iter = strings.iterator();
                sb.append(iter.next());
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    sb.append(iter.next());
                }
                break;
        }
        return sb;
    }

    static StringBuilder join(
        final List<String> strings,
        final char delimiter,
        final String prefix,
        final int overhead)
    {
        StringBuilder sb;
        int size = strings.size();
        switch (size) {
            case 0:
                sb = new StringBuilder(prefix.length() + overhead);
                sb.append(prefix);
                break;
            case 1:
                String str = strings.get(0);
                sb = new StringBuilder(
                    strlen(str) + prefix.length() + overhead);
                sb.append(prefix);
                sb.append(str);
                break;
            default:
                int length = prefix.length() + overhead + size - 1;
                for (int i = 0; i < size; ++i) {
                    length += strlen(strings.get(i));
                }
                sb = new StringBuilder(length);
                sb.append(prefix);
                sb.append(strings.get(0));
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    sb.append(strings.get(i));
                }
                break;
        }
        return sb;
    }

    static StringBuilder join(
        final List<String> strings,
        final String delimiter,
        final String prefix,
        final int overhead)
    {
        StringBuilder sb;
        int size = strings.size();
        switch (size) {
            case 0:
                sb = new StringBuilder(prefix.length() + overhead);
                sb.append(prefix);
                break;
            case 1:
                String str = strings.get(0);
                sb = new StringBuilder(
                    strlen(str) + prefix.length() + overhead);
                sb.append(prefix);
                sb.append(str);
                break;
            default:
                int length =
                    prefix.length() + overhead
                    + (size - 1) * delimiter.length();
                for (int i = 0; i < size; ++i) {
                    length += strlen(strings.get(i));
                }
                sb = new StringBuilder(length);
                sb.append(prefix);
                sb.append(strings.get(0));
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    sb.append(strings.get(i));
                }
                break;
        }
        return sb;
    }

    static StringBuilder join(
        final String[] strings,
        final char delimiter,
        final String prefix,
        final int overhead)
    {
        StringBuilder sb;
        int size = strings.length;
        switch (size) {
            case 0:
                sb = new StringBuilder(prefix.length() + overhead);
                sb.append(prefix);
                break;
            case 1:
                String str = strings[0];
                sb = new StringBuilder(
                    strlen(str) + prefix.length() + overhead);
                sb.append(prefix);
                sb.append(str);
                break;
            default:
                int length = prefix.length() + overhead + size - 1;
                for (String string: strings) {
                    length += strlen(string);
                }
                sb = new StringBuilder(length);
                sb.append(prefix);
                sb.append(strings[0]);
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    sb.append(strings[i]);
                }
                break;
        }
        return sb;
    }

    static StringBuilder join(
        final String[] strings,
        final String delimiter,
        final String prefix,
        final int overhead)
    {
        StringBuilder sb;
        int size = strings.length;
        switch (size) {
            case 0:
                sb = new StringBuilder(prefix.length() + overhead);
                sb.append(prefix);
                break;
            case 1:
                String str = strings[0];
                sb = new StringBuilder(
                    strlen(str) + prefix.length() + overhead);
                sb.append(prefix);
                sb.append(str);
                break;
            default:
                int length =
                    prefix.length() + overhead
                    + (size - 1) * delimiter.length();
                for (String string: strings) {
                    length += strlen(string);
                }
                sb = new StringBuilder(length);
                sb.append(prefix);
                sb.append(strings[0]);
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    sb.append(strings[i]);
                }
                break;
        }
        return sb;
    }

    static <T> String join(
        final Collection<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final char delimiter)
    {
        return new String(
            join(new StringBuilder(), values, appender, delimiter));
    }

    static <T> String join(
        final Collection<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final char delimiter,
        final String prefix,
        final String suffix)
    {
        return
            new String(
                join(new StringBuilder(prefix), values, appender, delimiter)
                    .append(suffix));
    }

    static <T> StringBuilder join(
        final StringBuilder sb,
        final Collection<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final char delimiter)
    {
        int size = values.size();
        switch (size) {
            case 0:
                break;
            case 1:
                sb.append(values.iterator().next());
                break;
            default:
                Iterator<? extends T> iter = values.iterator();
                appender.accept(sb, iter.next());
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    appender.accept(sb, iter.next());
                }
                break;
        }
        return sb;
    }

    static <T> String join(
        final Collection<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final String delimiter)
    {
        return new String(
            join(new StringBuilder(), values, appender, delimiter));
    }

    static <T> String join(
        final Collection<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final String delimiter,
        final String prefix,
        final String suffix)
    {
        return
            new String(
                join(new StringBuilder(prefix), values, appender, delimiter)
                    .append(suffix));
    }

    static <T> StringBuilder join(
        final StringBuilder sb,
        final Collection<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final String delimiter)
    {
        int size = values.size();
        switch (size) {
            case 0:
                break;
            case 1:
                sb.append(values.iterator().next());
                break;
            default:
                Iterator<? extends T> iter = values.iterator();
                appender.accept(sb, iter.next());
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    appender.accept(sb, iter.next());
                }
                break;
        }
        return sb;
    }

    static <T> String join(
        final List<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final char delimiter)
    {
        return new String(
            join(new StringBuilder(), values, appender, delimiter));
    }

    static <T> String join(
        final List<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final char delimiter,
        final String prefix,
        final String suffix)
    {
        return
            new String(
                join(new StringBuilder(prefix), values, appender, delimiter)
                    .append(suffix));
    }

    static <T> StringBuilder join(
        final StringBuilder sb,
        final List<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final char delimiter)
    {
        int size = values.size();
        switch (size) {
            case 0:
                break;
            case 1:
                sb.append(values.get(0));
                break;
            default:
                sb.append(values.get(0));
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    appender.accept(sb, values.get(i));
                }
                break;
        }
        return sb;
    }

    static <T> String join(
        final List<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final String delimiter)
    {
        return new String(
            join(new StringBuilder(), values, appender, delimiter));
    }

    static <T> String join(
        final List<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final String delimiter,
        final String prefix,
        final String suffix)
    {
        return
            new String(
                join(new StringBuilder(prefix), values, appender, delimiter)
                    .append(suffix));
    }

    static <T> StringBuilder join(
        final StringBuilder sb,
        final List<? extends T> values,
        final BiConsumer<StringBuilder, ? super T> appender,
        final String delimiter)
    {
        int size = values.size();
        switch (size) {
            case 0:
                break;
            case 1:
                sb.append(values.get(0));
                break;
            default:
                sb.append(values.get(0));
                for (int i = 1; i < size; ++i) {
                    sb.append(delimiter);
                    appender.accept(sb, values.get(i));
                }
                break;
        }
        return sb;
    }

    static String joinStringBuilderables(
        final Collection<? extends StringBuilderable> values,
        final char delimiter)
    {
        return join(values, StringBuilderableAppender.INSTANCE, delimiter);
    }

    static String joinStringBuilderables(
        final Collection<? extends StringBuilderable> values,
        final char delimiter,
        final String prefix,
        final String suffix)
    {
        return join(
            values,
            StringBuilderableAppender.INSTANCE,
            delimiter,
            prefix,
            suffix);
    }

    static StringBuilder joinStringBuilderables(
        final StringBuilder sb,
        final Collection<? extends StringBuilderable> values,
        final char delimiter)
    {
        return join(sb, values, StringBuilderableAppender.INSTANCE, delimiter);
    }

    static String joinStringBuilderables(
        final Collection<? extends StringBuilderable> values,
        final String delimiter)
    {
        return join(values, StringBuilderableAppender.INSTANCE, delimiter);
    }

    static String joinStringBuilderables(
        final Collection<? extends StringBuilderable> values,
        final String delimiter,
        final String prefix,
        final String suffix)
    {
        return join(
            values,
            StringBuilderableAppender.INSTANCE,
            delimiter,
            prefix,
            suffix);
    }

    static StringBuilder joinStringBuilderables(
        final StringBuilder sb,
        final Collection<? extends StringBuilderable> values,
        final String delimiter)
    {
        return join(sb, values, StringBuilderableAppender.INSTANCE, delimiter);
    }

    static String joinStringBuilderables(
        final List<? extends StringBuilderable> values,
        final char delimiter)
    {
        return join(values, StringBuilderableAppender.INSTANCE, delimiter);
    }

    static String joinStringBuilderables(
        final List<? extends StringBuilderable> values,
        final char delimiter,
        final String prefix,
        final String suffix)
    {
        return join(
            values,
            StringBuilderableAppender.INSTANCE,
            delimiter,
            prefix,
            suffix);
    }

    static StringBuilder joinStringBuilderables(
        final StringBuilder sb,
        final List<? extends StringBuilderable> values,
        final char delimiter)
    {
        return join(sb, values, StringBuilderableAppender.INSTANCE, delimiter);
    }

    static String joinStringBuilderables(
        final List<? extends StringBuilderable> values,
        final String delimiter)
    {
        return join(values, StringBuilderableAppender.INSTANCE, delimiter);
    }

    static String joinStringBuilderables(
        final List<? extends StringBuilderable> values,
        final String delimiter,
        final String prefix,
        final String suffix)
    {
        return join(
            values,
            StringBuilderableAppender.INSTANCE,
            delimiter,
            prefix,
            suffix);
    }

    static StringBuilder joinStringBuilderables(
        final StringBuilder sb,
        final List<? extends StringBuilderable> values,
        final String delimiter)
    {
        return join(sb, values, StringBuilderableAppender.INSTANCE, delimiter);
    }

    static void toStringBuilder(
        @Nonnull final StringBuilder sb,
        @Nullable final Collection<String> values)
    {
        if (values == null) {
            sb.append("null");
        } else {
            boolean empty = true;
            sb.append('[');
            for (String value: values) {
                if (empty) {
                    empty = false;
                } else {
                    sb.append(',');
                }
                sb.append(value);
            }
            sb.append(']');
        }
    }

    static int updateHashCode(final int hashCode, final char c) {
        return hashCode * STRING_HASH_CODE_MULTIPLIER + c;
    }

    static int updateHashCode(final int hashCode, final String str) {
        int hash = hashCode;
        for (int i = str.length(); i-- > 0; ) {
            hash *= STRING_HASH_CODE_MULTIPLIER;
        }
        return hash + str.hashCode();
    }

    static String trimSequences(
        final String str,
        final IntPredicate predicate,
        final int maxSequenceLen)
    {
        int len = str.length();
        int off = 0;
        StringBuilder sb = null;
        while (true) {
            int start = off;
            while (start < len) {
                if (predicate.test(str.charAt(start))) {
                    break;
                } else {
                    ++start;
                }
            }
            if (start == len) {
                if (sb == null) {
                    return str;
                } else {
                    sb.append(str, off, len);
                    return new String(sb);
                }
            }
            int sequenceStart = start;
            int sequenceLen = 1;
            int pos = start + 1;
            while (pos < len) {
                if (predicate.test(str.charAt(pos))) {
                    ++pos;
                    ++sequenceLen;
                } else if (sequenceLen > maxSequenceLen) {
                    break;
                } else {
                    while (pos < len) {
                        if (predicate.test(str.charAt(pos))) {
                            sequenceStart = pos++;
                            sequenceLen = 1;
                            break;
                        } else {
                            ++pos;
                        }
                    }
                }
            }
            if (sequenceLen <= maxSequenceLen) {
                if (sb == null) {
                    return str;
                } else {
                    sb.append(str, off, len);
                    return new String(sb);
                }
            }
            if (pos == len) {
                if (sb == null) {
                    return str.substring(off, sequenceStart + maxSequenceLen);
                } else {
                    sb.append(str, off, sequenceStart + maxSequenceLen);
                    return new String(sb);
                }
            }
            if (sb == null) {
                sb = new StringBuilder(len + maxSequenceLen - sequenceLen);
            }
            sb.append(str, off, sequenceStart + maxSequenceLen);
            off = pos;
        }
    }
}

