package ru.yandex.util.system;

import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.lang.management.RuntimeMXBean;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.RoundingMode;

public class CPUMonitor {
    private static final String GET_PROCESS_CPU_TIME = "getProcessCpuTime";
    private static final String GET_SYSTEM_CPU_LOAD = "getSystemCpuLoad";
    private static final double ONE_HUNDRED = 100d;
    private static final double SUN_DIVISOR = 10000d;
    private static final double IBM_DIVISOR = 100d;
    private static final double MIN_CPU_TIME_UPDATE_INTERVAL = 1000d;
    private static final int MILLIS_IN_SECS = 1000;

    private final OperatingSystemMXBean osBean;
    private final RuntimeMXBean runtimeBean;
    private final Method getProcessCpuTime;
    private final Method getSystemCpuLoad;
    private final int cpuCount;
    private final double divisor;
    private long prevProcessCpuTime = -1L;
    private long prevUpTime = -1L;

    public CPUMonitor() throws ReflectiveOperationException {
        Class<?> sunOsBean;
        try {
            sunOsBean =
                ClassLoader.getSystemClassLoader().loadClass(
                    "com.sun.management.OperatingSystemMXBean");
        } catch (ReflectiveOperationException e) {
            sunOsBean = null;
        }
        Class<?> ibmOsBean;
        try {
            ibmOsBean =
                ClassLoader.getSystemClassLoader().loadClass(
                    "com.ibm.lang.management.OperatingSystemMXBean");
        } catch (ReflectiveOperationException e) {
            ibmOsBean = null;
        }

        osBean = ManagementFactory.getOperatingSystemMXBean();
        if (sunOsBean != null && sunOsBean.isInstance(osBean)) {
            getProcessCpuTime = getMethod(sunOsBean, GET_PROCESS_CPU_TIME);
            getSystemCpuLoad = getMethod(sunOsBean, GET_SYSTEM_CPU_LOAD);
            divisor = SUN_DIVISOR;
        } else if (ibmOsBean != null && ibmOsBean.isInstance(osBean)) {
            getProcessCpuTime = getMethod(ibmOsBean, GET_PROCESS_CPU_TIME);
            getSystemCpuLoad = getMethod(ibmOsBean, GET_SYSTEM_CPU_LOAD);
            divisor = IBM_DIVISOR;
        } else {
            throw new ReflectiveOperationException(
                "Unhandled OperatingSystemMXBean class type: "
                    + osBean.getClass().getName());
        }
        runtimeBean = ManagementFactory.getRuntimeMXBean();
        cpuCount = Runtime.getRuntime().availableProcessors();
    }

    private static Method getMethod(final Class<?> clazz, final String name) {
        try {
            return clazz.getMethod(name);
        } catch (ReflectiveOperationException e) {
            return null;
        }
    }

    public int cpuCount() {
        return cpuCount;
    }

    public long getProcessCpuTime() throws ReflectiveOperationException {
        if (getProcessCpuTime == null) {
            return 0L;
        } else {
            return (Long) getProcessCpuTime.invoke(osBean);
        }
    }

    public static double round(final double value, final int places) {
        if (places < 0) {
            throw new IllegalArgumentException();
        }
        BigDecimal bd = new BigDecimal(value);
        bd = bd.setScale(places, RoundingMode.HALF_UP);
        return bd.doubleValue();
    }

    public double getCpuUsage() {
        try {
            final long upTime = runtimeBean.getUptime();
            final long processCpuTime = getProcessCpuTime();
            final double elapsedTime = (double) (upTime - prevUpTime);
            final double elapsedCpu =
                (double) (processCpuTime - prevProcessCpuTime);
            final double cpuUsage =
                Math.max(
                    Math.min(
                        elapsedCpu / (elapsedTime * divisor),
                        (cpuCount * ONE_HUNDRED) - 1),
                    0);
            if (elapsedTime > MIN_CPU_TIME_UPDATE_INTERVAL) {
                prevUpTime = upTime;
                prevProcessCpuTime = processCpuTime;
            }
            return round(cpuUsage, 2);
        } catch (ReflectiveOperationException e) {
            return -1;
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    public double getSystemCpuLoad() {
        double load;
        if (getSystemCpuLoad == null) {
            load = -1;
        } else {
            try {
                load = (double) getSystemCpuLoad.invoke(osBean);
                load *= ONE_HUNDRED * cpuCount;
            } catch (ReflectiveOperationException e) {
                load = -1;
            }
        }
        return Math.max(0, round(load, 2));
    }

    public long getUpTimeSecs() {
        return runtimeBean.getUptime() / MILLIS_IN_SECS;
    }
}
