package ru.yandex.util.system;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ThreadFactory;
import java.util.function.Consumer;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.util.timesource.TimeSource;

public class RusageMonitor
    implements GenericAutoCloseable<RuntimeException>, Runnable
{
    private final Queue<Record> records = new ConcurrentLinkedQueue<>();
    private final long maxAge;
    private final long timerResolution;
    private final Consumer<? super RusageException> errorHandler;
    private final Thread thread;
    private volatile boolean closed = false;

    public RusageMonitor(
        final long maxAge,
        final long timerResolution,
        final Consumer<? super RusageException> errorHandler,
        final ThreadFactory threadFactory)
    {
        this.maxAge = maxAge;
        this.timerResolution = timerResolution;
        this.errorHandler = errorHandler;
        thread = threadFactory.newThread(this);
    }

    public void start() {
        thread.start();
    }

    @Override
    public void close() {
        closed = true;
        thread.interrupt();
    }

    @Override
    public void run() {
        while (!closed) {
            try {
                update();
            } catch (RusageException e) {
                errorHandler.accept(e);
            }
            try {
                Thread.sleep(timerResolution);
            } catch (InterruptedException e) {
                break;
            }
        }
    }

    private void update() throws RusageException {
        Iterator<Record> iter = records.iterator();
        long now = TimeSource.INSTANCE.currentTimeMillis();
        long minTimestamp = now - maxAge;
        while (iter.hasNext()) {
            Record record = iter.next();
            if (record.timestamp < minTimestamp) {
                iter.remove();
            } else {
                break;
            }
        }
        records.add(new Record(Rusage.get(), now));
    }

    public Snapshot snapshot() {
        return snapshot(Long.MAX_VALUE);
    }

    public Snapshot snapshot(final long interval) {
        Record startRecord = null;
        Record endRecord = null;
        Record peakStartRecord = Record.NULL;
        Record peakEndRecord = Record.NULL;
        double peakUsage = 0d;
        long minTimestamp = TimeSource.INSTANCE.currentTimeMillis() - interval;
        for (Record record: records) {
            if (record.timestamp >= minTimestamp) {
                if (startRecord == null) {
                    startRecord = record;
                }
                if (endRecord != null) {
                    double usage =
                        userUsage(endRecord, record)
                        + systemUsage(endRecord, record);
                    if (usage > peakUsage) {
                        peakUsage = usage;
                        peakStartRecord = endRecord;
                        peakEndRecord = record;
                    }
                }
                endRecord = record;
            }
        }
        if (startRecord == null) {
            return Snapshot.NULL;
        } else {
            return new Snapshot(
                startRecord,
                endRecord,
                peakStartRecord,
                peakEndRecord);
        }
    }

    public static double userUsage(
        final Record startRecord,
        final Record endRecord)
    {
        long diff = endRecord.timestamp - startRecord.timestamp;
        if (diff <= 0L) {
            return 0d;
        } else {
            double divisor = diff * 10L;
            long usage =
                endRecord.rusage.userTime()
                - startRecord.rusage.userTime();
            return usage / divisor;
        }
    }

    public static double systemUsage(
        final Record startRecord,
        final Record endRecord)
    {
        long diff = endRecord.timestamp - startRecord.timestamp;
        if (diff <= 0L) {
            return 0d;
        } else {
            double divisor = diff * 10L;
            long usage =
                endRecord.rusage.systemTime()
                - startRecord.rusage.systemTime();
            return usage / divisor;
        }
    }

    public static double round(
        final double value, final int decimalPoints)
    {
        BigDecimal bd = new BigDecimal(value);
        bd = bd.setScale(decimalPoints, RoundingMode.HALF_UP);
        return bd.doubleValue();
    }

    public static class Record {
        public static final Record NULL = new Record(Rusage.NULL, 0L);
        private final Rusage rusage;
        private final long timestamp;

        Record(final Rusage rusage, final long timestamp) {
            this.rusage = rusage;
            this.timestamp = timestamp;
        }

        public Rusage rusage() {
            return rusage;
        }

        public long timestamp() {
            return timestamp;
        }
    }

    public static class Snapshot {
        public static final Snapshot NULL =
            new Snapshot(Record.NULL, Record.NULL, Record.NULL, Record.NULL);

        private final Record startRecord;
        private final Record endRecord;
        private final Record peakStartRecord;
        private final Record peakEndRecord;

        public Snapshot(
            final Record startRecord,
            final Record endRecord,
            final Record peakStartRecord,
            final Record peakEndRecord)
        {
            this.startRecord = startRecord;
            this.endRecord = endRecord;
            this.peakStartRecord = peakStartRecord;
            this.peakEndRecord = peakEndRecord;
        }

        public Record startRecord() {
            return startRecord;
        }

        public Record endRecord() {
            return endRecord;
        }

        public Record peakStartRecord() {
            return peakStartRecord;
        }

        public Record peakEndRecord() {
            return peakEndRecord;
        }

        public double totalCPUUsage() {
            return RusageMonitor.userUsage(startRecord, endRecord)
                + RusageMonitor.systemUsage(startRecord, endRecord);
        }

        public double peakCPUUsage() {
            return RusageMonitor.userUsage(peakStartRecord, peakEndRecord)
                + RusageMonitor.systemUsage(peakStartRecord, peakEndRecord);
        }
    }
}

