package ru.yandex.unsafe;

import java.lang.reflect.Field;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;

import sun.misc.Unsafe;

public final class NativeMemory2 {
    public static final int LONG_SHIFT = 3;
    public static final int INT_SHIFT = 2;
    public static final int SHORT_SHIFT = 1;

    private static final int MAX_POOLED_BUFFER_SIZE = 1024 * 1024;
    private static final int BUFFER_SIZE_STEP_SHIFT = 8;
    private static final boolean DEBUG = true;

    private static final AtomicReferenceArray<
        ConcurrentLinkedQueue<Long>> SIZE_POOLS =
        new AtomicReferenceArray<>(
            (MAX_POOLED_BUFFER_SIZE >> BUFFER_SIZE_STEP_SHIFT) + 1);

    private static final AtomicLong TOTAL_SIZE = new AtomicLong();
    private static final ConcurrentMap<String, NativeMemoryAllocator> POOLS =
        new ConcurrentHashMap<>();

    private static final NativeMemoryAllocator POOLED_ALLOCATOR =
        NativeMemoryAllocator.get("POOLED");

    private static final int UNBOXED_HEAD_SIZE = 4;
    private static final int PRINT_STATS_DELAY = 5000;
    private static final int VINT_NEXT_BYTE_MASK = 0x80;
    private static final int VINT_VALUE_MASK = 0x7f;
    private static final long VLONG_VALUE_MASK = 0x7fL;
    private static final int VINT_SHIFT = 7;

    private static final Unsafe UNSAFE = getUnsafe();

    private static long previousStats = -1;

    private final NativeMemoryAllocator allocator;
    private long address;
    private long size;
    private volatile boolean closed = false;

    private NativeMemory2(
        final long address,
        final long size,
        final NativeMemoryAllocator allocator)
    {
        this.address = address;
        if (address == 0) {
            throw new RuntimeException("NPE Address = 0, out of memory, size="
                + size);
        }
        this.size = size;
        this.allocator = allocator;
        allocator.allocs.incrementAndGet();
        allocator.privateSize.addAndGet(size);
        TOTAL_SIZE.addAndGet(size);
    }

    public NativeMemoryAllocator allocator() {
        return allocator;
    }

    private static int poolIndex(final int size) {
        final int index;
        final int div = size >> BUFFER_SIZE_STEP_SHIFT;
        final int reminder = size - (div << BUFFER_SIZE_STEP_SHIFT);
        if (reminder == 0) {
            index = div - 1;
        } else {
            index = div;
        }
        return index;
    }

    private static long getPooledUnboxed(
        final NativeMemoryAllocator allocator,
        final int size)
    {
        if (size > MAX_POOLED_BUFFER_SIZE) {
            return allocator.doAllocUnboxed(size);
        }
        final int index = poolIndex(size);
        final int nextSize = (index + 1) << BUFFER_SIZE_STEP_SHIFT;
        final long ret;
        if (index >= SIZE_POOLS.length()) {
            new ArrayIndexOutOfBoundsException("AIOOB in getBuffer: size="
                + size + ", gindex=" + index
                + ", gnextSize=" + nextSize).printStackTrace();
            ret = allocator.doAllocUnboxed(nextSize);
        } else {
            ConcurrentLinkedQueue<Long> pool = SIZE_POOLS.get(index);
            if (pool == null) {
                pool = new ConcurrentLinkedQueue<>();
                if (!SIZE_POOLS.compareAndSet(index, null, pool)) {
                    pool = SIZE_POOLS.get(index);
                }
            }
            Long pooled = pool.poll();
            if (pooled == null) {
                ret = allocator.doAllocUnboxed(nextSize);
            } else {
                ret = pooled;
                POOLED_ALLOCATOR.privateSize.addAndGet(-unboxedSize(ret));
            }
        }
        return ret;
    }

    private static void freePooledUnboxed(
        final NativeMemoryAllocator allocator,
        final long pooled)
    {
        final int size = unboxedSize(pooled);
        if (size > MAX_POOLED_BUFFER_SIZE) {
            allocator.doFreeUnboxed(pooled);
        }
        final int index = poolIndex(size);
        final int nextSize = (index + 1) << BUFFER_SIZE_STEP_SHIFT;
        if (index >= SIZE_POOLS.length()) {
            new ArrayIndexOutOfBoundsException("AIOOB in freeBuffer: size="
                + size + ", index=" + index
                + ", nextSize=" + nextSize).printStackTrace();
        } else {
            ConcurrentLinkedQueue<Long> pool = SIZE_POOLS.get(index);
            if (pool == null) {
                pool = new ConcurrentLinkedQueue<>();
                if (!SIZE_POOLS.compareAndSet(index, null, pool)) {
                    pool = SIZE_POOLS.get(index);
                }
            }
            pool.offer(pooled);
        }
//        allocator.privateSize.addAndGet(-size);
        POOLED_ALLOCATOR.privateSize.addAndGet(size);
    }

    private static Unsafe getUnsafe() {
        try {
            Field theUnsafe = Unsafe.class.getDeclaredField("theUnsafe");
            theUnsafe.setAccessible(true);
            Unsafe unsafe = (Unsafe) theUnsafe.get(null);
            Thread t = new Thread("NativeMemoryStatsPrinter") {
                @Override
                public void run() {
                    while (true) {
                        try {
                            Thread.sleep(PRINT_STATS_DELAY);
                            printStats();
                        } catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                    }
                }
            };
            t.setDaemon(true);
            t.start();
            return unsafe;
        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    public static final class NativeMemoryAllocator {
        private final AtomicLong privateSize = new AtomicLong();
        private final AtomicLong allocs = new AtomicLong();
        private final AtomicLong reallocs = new AtomicLong();
        private final AtomicLong deallocs = new AtomicLong();
        private final boolean pool;

        private NativeMemoryAllocator(final boolean pool) {
            this.pool = pool;
        }

        public static NativeMemoryAllocator get(final String name) {
            NativeMemoryAllocator allocator = POOLS.get(name);
            if (allocator == null) {
                allocator = new NativeMemoryAllocator(false);
                NativeMemoryAllocator old = POOLS.putIfAbsent(name, allocator);
                if (old != null) {
                    allocator = old;
                }
            }
            return allocator;
        }

        public static Iterator<Map.Entry<String, NativeMemoryAllocator>>
            iterator()
        {
            return POOLS.entrySet().iterator();
        }

        public static NativeMemoryAllocator getPooled(final String name) {
            NativeMemoryAllocator allocator = POOLS.get(name);
            if (allocator == null) {
                allocator = new NativeMemoryAllocator(true);
                NativeMemoryAllocator old = POOLS.putIfAbsent(name, allocator);
                if (old != null) {
                    allocator = old;
                }
            }
            return allocator;
        }

        public NativeMemory2 alloc(final long size) {
            return new NativeMemory2(UNSAFE.allocateMemory(size), size, this);
        }

        public long allocUnboxed(final int size) {
            if (pool) {
                return getPooledUnboxed(this, size);
            } else {
                return doAllocUnboxed(size);
            }
        }

        public long reallocUnboxed(final long address, final int size) {
            final long oldAddress = address - UNBOXED_HEAD_SIZE;
            final int unboxedSize =
                NativeMemory2.unboxedGetInt(oldAddress);
            int diff = size - unboxedSize;
            long newAddress = UNSAFE.reallocateMemory(
                oldAddress,
                size + UNBOXED_HEAD_SIZE);
            if (newAddress == 0) {
                throw new RuntimeException(
                    "NPE ReallocUnboxed ouf of memory: newAddress = 0, "
                    + "oldSize=" + unboxedSize + ", newSize=" + size);
            }
            privateSize.addAndGet(diff);
            TOTAL_SIZE.addAndGet(diff);
            reallocs.incrementAndGet();
            NativeMemory2.unboxedSetInt(newAddress, size);
            return newAddress + UNBOXED_HEAD_SIZE;
        }

        public long allocLongUnboxed(final int longSize) {
            final int size = longSize << LONG_SHIFT;
            if (pool) {
                return getPooledUnboxed(this, size);
            } else {
                return doAllocUnboxed(size);
            }
        }

        public long doAllocUnboxed(final int size) {
            final int unboxedSize = size + UNBOXED_HEAD_SIZE;
            final long address = UNSAFE.allocateMemory(unboxedSize);
            if (address == 0) {
                throw new RuntimeException("NPE out of memory: address = 0, "
                    + "size=" + size);
            }
            allocs.incrementAndGet();
            privateSize.addAndGet(unboxedSize);
            TOTAL_SIZE.addAndGet(unboxedSize);
            NativeMemory2.unboxedSetInt(address, size);
            return address + UNBOXED_HEAD_SIZE;
        }

        public void freeUnboxed(final long address) {
            if (pool) {
                freePooledUnboxed(this, address);
            } else {
                doFreeUnboxed(address);
            }
        }

        public void doFreeUnboxed(final long address) {
            final int size =
                NativeMemory2.unboxedGetInt(address - UNBOXED_HEAD_SIZE)
                + UNBOXED_HEAD_SIZE;
            TOTAL_SIZE.addAndGet(-size);
            privateSize.addAndGet(-size);
            deallocs.incrementAndGet();
            UNSAFE.freeMemory(address - UNBOXED_HEAD_SIZE);
        }

        public NativeMemory2 allocLong(final long longSize) {
            final long size = longSize << LONG_SHIFT;
            return new NativeMemory2(UNSAFE.allocateMemory(size), size, this);
        }

        public NativeMemory2 allocInt(final long intSize) {
            final long size = intSize << INT_SHIFT;
            return new NativeMemory2(UNSAFE.allocateMemory(size), size, this);
        }

        public long memorySize() {
            return privateSize.get();
        }
    }

    public NativeMemory2 realloc(final long newSize) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        address = UNSAFE.reallocateMemory(address, newSize);
        if (address == 0) {
            throw new RuntimeException("NPE Realloc failed: address = 0"
                + ", oldSize=" + size + ", newSize=" + newSize);
        }
        long diff = newSize - size;
        TOTAL_SIZE.addAndGet(diff);
        allocator.privateSize.addAndGet(diff);
        allocator.reallocs.incrementAndGet();
        size = newSize;
        return this;
    }

    public synchronized void free() {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        if (address != 0) {
            TOTAL_SIZE.addAndGet(-size);
            allocator.privateSize.addAndGet(-size);
            allocator.deallocs.incrementAndGet();
            UNSAFE.freeMemory(address);
            address = 0;
            closed = true;
        } else {
            new Exception("NativeMemory double free").printStackTrace();
        }
    }

    public long address() {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return address;
    }

    public void write(final byte[] src) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.copyMemory(
            src,
            Unsafe.ARRAY_BYTE_BASE_OFFSET,
            null,
            address,
            src.length);
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
    }

    public void write(
        final long offset,
        final byte[] src,
        final int srcOffset,
        final int length)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.copyMemory(
            src,
            srcOffset + Unsafe.ARRAY_BYTE_BASE_OFFSET,
            null,
            address + offset,
            length);
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
    }

    public void write(
        final long offset,
        final NativeMemory2 src,
        final int srcOffset,
        final int length)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.copyMemory(src.address + srcOffset, address + offset, length);
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
    }

    public void writeSlow(
        final long offset,
        final byte[] src,
        final int srcOffset,
        final int length)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        long pos = offset + address;
        final long stop = pos + length;
        int inPos = srcOffset;
        while (pos < stop) {
            UNSAFE.putByte(pos++, src[inPos++]);
        }
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
    }

    public static void unboxedWrite(final long address, final byte[] src) {
        UNSAFE.copyMemory(
            src,
            Unsafe.ARRAY_BYTE_BASE_OFFSET,
            null,
            address,
            src.length);
    }

    public static void unboxedWrite(
        final long address,
        final NativeMemory2 src,
        final int srcOffset,
        final int length)
    {
        UNSAFE.copyMemory(
            src.address + srcOffset,
            address,
            length);
    }

    public static void unboxedWrite(
        final long address,
        final byte[] src,
        final int srcOffset,
        final int length)
    {
        UNSAFE.copyMemory(
            src,
            srcOffset + Unsafe.ARRAY_BYTE_BASE_OFFSET,
            null,
            address,
            length);
    }

    public static void unboxedWriteSlow(
        final long address,
        final byte[] src,
        final int srcOffset,
        final int length)
    {
        long pos = address;
        final long stop = pos + length;
        int inPos = srcOffset;
        while (pos < stop) {
            UNSAFE.putByte(pos++, src[inPos++]);
        }
    }

    public void read(
        final long offset,
        final byte[] dst,
        final int dstOffset,
        final int length)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.copyMemory(
            null,
            address + offset,
            dst,
            dstOffset + Unsafe.ARRAY_BYTE_BASE_OFFSET,
            length);
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
    }

    public void read(
        final long offset,
        final NativeMemory2 dst,
        final int dstOffset,
        final int length)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.copyMemory(address + offset, dst.address + dstOffset, length);
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
    }

    public void read(
        final long offset,
        final long dst,
        final int length)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.copyMemory(address + offset, dst, length);
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
    }

    public void readSlow(
        final long offset,
        final byte[] dst,
        final int dstOffset,
        final int length)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        long pos = offset + address;
        final long stop = pos + length;
        int outPos = dstOffset;
        while (pos < stop) {
            dst[outPos++] = UNSAFE.getByte(pos++);
        }
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
    }

    public static void unboxedRead(
        final long address,
        final byte[] dst,
        final int dstOffset,
        final int length)
    {
        UNSAFE.copyMemory(
            null,
            address,
            dst,
            dstOffset + Unsafe.ARRAY_BYTE_BASE_OFFSET,
            length);
    }

    public static void unboxedReadSlow(
        final long address,
        final byte[] dst,
        final int dstOffset,
        final int length)
    {
        long pos = address;
        final long stop = pos + length;
        int outPos = dstOffset;
        while (pos < stop) {
            dst[outPos++] = UNSAFE.getByte(pos++);
        }
    }

    public int size() {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return (int) size;
    }

    public int longSize() {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return (int) size >> LONG_SHIFT;
    }

    public int intSize() {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return (int) size >> INT_SHIFT;
    }

    public static int unboxedSize(final long address) {
        return unboxedGetInt(address - UNBOXED_HEAD_SIZE);
    }

    public long getLong(final long offset) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return UNSAFE.getLong(address + (offset << LONG_SHIFT));
    }

    public void setLong(final long offset, final long value) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putLong(address + (offset << LONG_SHIFT), value);
    }

    public void setLongUnaligned(final long offset, final long value) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putLong(address + offset, value);
    }

    public void swapLongUnaligned(final long offset1, final long offset2) {
        final long o1 = address + offset1;
        final long o2 = address + offset2;
        final long tmp = UNSAFE.getLong(o1);
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putLong(o1, UNSAFE.getLong(o2));
        UNSAFE.putLong(o2, tmp);
    }

    public long getLongUnaligned(final long offset) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return UNSAFE.getLong(address + offset);
    }

    public void setIntUnaligned(final long offset, final int value) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putInt(address + offset, value);
    }

    public int getIntUnaligned(final long offset) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return UNSAFE.getInt(address + offset);
    }

    public void swapIntUnaligned(final long offset1, final long offset2) {
        final long o1 = address + offset1;
        final long o2 = address + offset2;
        final int tmp = UNSAFE.getInt(o1);
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putInt(o1, UNSAFE.getInt(o2));
        UNSAFE.putInt(o2, tmp);
    }

    public static void unboxedSetLong(final long address, final long value) {
        UNSAFE.putLong(address, value);
    }

    public long getLongVolatile(final long offset) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return UNSAFE.getLongVolatile(null, address + (offset << LONG_SHIFT));
    }

    public void putLongVolatile(final long offset, final long value) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putLongVolatile(null, address + (offset << LONG_SHIFT), value);
    }

    public boolean compareAndSetLong(
        final long offset,
        final long expect,
        final long update)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return UNSAFE.compareAndSwapLong(
            null,
            address + (offset << LONG_SHIFT),
            expect,
            update);
    }

    public void andLong(final long offset, final long mask) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        final long address = this.address + (offset << LONG_SHIFT);
        UNSAFE.putLong(address, UNSAFE.getLong(address) & mask);
    }

    public void orLong(final long offset, final long mask) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        final long address = this.address + (offset << LONG_SHIFT);
        UNSAFE.putLong(address, UNSAFE.getLong(address) | mask);
    }

    public int getInt(final long offset) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return UNSAFE.getInt(address + (offset << INT_SHIFT));
    }

    public void setInt(final long offset, final int value) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putInt(address + (offset << INT_SHIFT), value);
    }

    public static int unboxedGetInt(final long address) {
        return UNSAFE.getInt(address);
    }

    public static void unboxedSetInt(final long address, final int value) {
        UNSAFE.putInt(address, value);
    }

    public short getShort(final long offset) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return UNSAFE.getShort(address + (offset << SHORT_SHIFT));
    }

    public void setShort(final long offset, final short value) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putShort(address + (offset << SHORT_SHIFT), value);
    }

    public void setByte(final long offset, final byte value) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putByte(offset + address, value);
    }

    public static void unboxedSetByte(final long address, final byte value) {
        UNSAFE.putByte(address, value);
    }

    public byte getByte(final long offset) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return UNSAFE.getByte(offset + address);
    }

    public static byte unboxedGetByte(final long address) {
        return UNSAFE.getByte(address);
    }

    public static void unboxedCopy(
        final long src,
        final long dst,
        final int size)
    {
        UNSAFE.copyMemory(src, dst, size);
    }

    public static int unboxedGetVInt(final long address, final int[] offset) {
        long pos = address + offset[0];
        byte b = UNSAFE.getByte(pos++);
        offset[0]++;
        int i = b & VINT_VALUE_MASK;
        for (int shift = VINT_SHIFT; (b & VINT_NEXT_BYTE_MASK) != 0;
            shift += VINT_SHIFT)
        {
            b = UNSAFE.getByte(pos++);
            offset[0]++;
            i |= (b & VINT_VALUE_MASK) << shift;
        }
        return i;
    }

    public static long unboxedGetVLong(final long address, final int[] offset) {
        long pos = address + offset[0];
        byte b = UNSAFE.getByte(pos++);
        offset[0]++;
        long i = b & VLONG_VALUE_MASK;
        for (int shift = VINT_SHIFT; (b & VINT_NEXT_BYTE_MASK) != 0;
            shift += VINT_SHIFT)
        {
            b = UNSAFE.getByte(pos++);
            offset[0]++;
            i |= (b & VLONG_VALUE_MASK) << shift;
        }
        return i;
    }

    public void putByte(final long offset, final byte b) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.putByte(offset + address, b);
    }

    @Override
    public int hashCode() {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        return super.hashCode();
    }

    @Override
    public boolean equals(final Object o) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        if (o == null || !(o instanceof NativeMemory2)) {
            return false;
        }
        boolean equals = false;
        NativeMemory2 other = (NativeMemory2) o;
        if (size() == other.size()) {
            equals = true;
            for (long i = 0; i < size; i++) {
                if (other.getByte(i) != getByte(i)) {
                    equals = false;
                    break;
                }
            }
        }
        return equals;
    }

    public static boolean unboxedEquals(
        final long address1,
        final long address2)
    {
        boolean equals = false;
        final int size1 = unboxedSize(address1);
        final int size2 = unboxedSize(address2);
        if (size1 == size2) {
            long pos1 = address1;
            long pos2 = address2;
            long end = address1 + size1;
            equals = true;
            for (; pos1 < end; pos1++, pos2++) {
                if (UNSAFE.getByte(pos1) != UNSAFE.getByte(pos2)) {
                    equals = false;
                    break;
                }
            }
        }
        return equals;
    }

    public NativeMemory2 fill(final byte value) {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.setMemory(address, size, value);
        return this;
    }

    public void fill(
        final long fromOffset,
        final long toOffset,
        final byte value)
    {
        if (DEBUG) {
            if (closed) {
                throw new RuntimeException("Operating on freed memory");
            }
        }
        UNSAFE.setMemory(address + fromOffset, toOffset - fromOffset, value);
    }

    public static void unboxedFill(final long address, final byte value) {
        final int size = unboxedGetInt(address - UNBOXED_HEAD_SIZE);
        UNSAFE.setMemory(address, size, value);
    }

    public static void printStats() {
        StringBuilder sb = new StringBuilder();
        long stats = TOTAL_SIZE.get();
        sb.append("\n\tNativeMemory Stats:");
        sb.append('\n');
        sb.append("\ttotalSize");
        sb.append(':');
        sb.append(' ');
        sb.append(stats);
        sb.append('\n');
        for (Map.Entry<String, NativeMemoryAllocator> entry
            : POOLS.entrySet())
        {
            sb.append('\t');
            sb.append(entry.getKey());
            sb.append(':');
            sb.append(' ');
            long tmp = entry.getValue().privateSize.get();
            stats += tmp;
            sb.append(tmp);
            sb.append(' ');
            sb.append(entry.getValue().allocs.get());
            sb.append('/');
            sb.append(entry.getValue().reallocs.get());
            sb.append('/');
            sb.append(entry.getValue().deallocs.get());
            sb.append('\n');
        }
        if (stats == previousStats) {
            return;
        }
        previousStats = stats;
        System.err.println(sb.toString());
    }
}
