#include <internal/module.h>
#include <iostream>
#include <openssl/ssl.h>

using namespace v8;

namespace nodejs {

v8::Persistent<v8::Function> Vdirect::constructor;

Vdirect::Vdirect(Isolate* isolate, const std::string& path) {
    try {
        static bool sslNeedInit = true;
        if (sslNeedInit) {
            SSL_library_init();
            SSL_load_error_strings();

            sslNeedInit = false;
        }

        storage = getKeysStorage(path);
    }
    catch (const std::exception& e) {
        const std::string msg = std::string("Unexpected exception in vdirect_nodejs: ") + e.what();
        isolate->ThrowException(Exception::Error(String::NewFromUtf8(isolate, msg.c_str())));
    }
}

bool checkParams(int expectedSize, Isolate* isolate, const v8::FunctionCallbackInfo<v8::Value>& args) {
    if (args.Length() != expectedSize) {
        isolate->ThrowException(Exception::TypeError(String::NewFromUtf8(isolate, "Wrong number of arguments")));
        return false;
    }

    for (int i = 0; i < expectedSize; i++) {
        if (args[i]->IsUndefined() || !args[i]->IsString()) {
            isolate->ThrowException(Exception::TypeError(String::NewFromUtf8(isolate, "Argument must be a string")));
            return false;
        }
    }

    return true;
}

void Vdirect::New(const v8::FunctionCallbackInfo<Value>& args) {
    Isolate* isolate = args.GetIsolate();

    if (!checkParams(1, isolate, args)) {
        return;
    }

    if (args.IsConstructCall()) {
        Vdirect* vdir = new Vdirect(isolate, to_string(args[0]));
        vdir->Wrap(args.This());
        args.GetReturnValue().Set(args.This());
    } else {
        const int argc = 1;
        Local<Value> argv[argc] = { args[0] };
        Local<Function> cons = Local<Function>::New(isolate, constructor);
        args.GetReturnValue().Set(cons->NewInstance(argc, argv));
    }
}

void Vdirect::createHashForUidLink(const v8::FunctionCallbackInfo<v8::Value>& args) {
    Isolate* isolate = args.GetIsolate();

    if (!checkParams(2, isolate, args)) {
        return;
    }

    try {
        std::string uid = to_string(args[0]);
        std::string link = to_string(args[1]);

        Vdirect* obj = ObjectWrap::Unwrap<Vdirect>(args.Holder());
        std::string hash = getValidator(obj->storage, uid).hash(link);

        args.GetReturnValue().Set(String::NewFromUtf8(isolate, hash.c_str()));
    } catch (const std::exception& e) {
        const std::string msg = std::string("Unexpected exception in vdirect_nodejs: ") + e.what();
        isolate->ThrowException(Exception::Error(String::NewFromUtf8(isolate, msg.c_str())));
    } catch (...) {
        isolate->ThrowException(Exception::Error(String::NewFromUtf8(isolate,
                                                                     "Unexpected exception in vdirect_nodejs")));
    }
}

void Vdirect::validateHashForUidLink(const v8::FunctionCallbackInfo<Value>& args) {
    Isolate* isolate = args.GetIsolate();

    if (!checkParams(3, isolate, args)) {
        return;
    }

    try {
        std::string uid = to_string(args[0]);
        std::string url = to_string(args[1]);
        std::string hash = to_string(args[2]);

        Vdirect* obj = ObjectWrap::Unwrap<Vdirect>(args.Holder());
        const bool valid = getValidator(obj->storage, uid).valid(url, hash);

        args.GetReturnValue().Set(Boolean::New(isolate, valid));
    } catch (...) {
        args.GetReturnValue().Set(Boolean::New(isolate, false));
    }
}

void Vdirect::validateHashForSmsLink(const v8::FunctionCallbackInfo<v8::Value>& args) {
    Isolate* isolate = args.GetIsolate();

    const int paramsCount = 3;
    if (args.Length() != paramsCount) {
        isolate->ThrowException(Exception::TypeError(String::NewFromUtf8(isolate, "Wrong number of arguments")));
        return;
    }

    for (int i = 0; i < paramsCount-1; i++) {
        if (args[i]->IsUndefined() || !args[i]->IsString()) {
            isolate->ThrowException(Exception::TypeError(String::NewFromUtf8(isolate, "Argument must be a string")));
            return;
        }
    }

    if (args[paramsCount-1]->IsUndefined() || !args[paramsCount-1]->IsNumber()) {
        isolate->ThrowException(Exception::TypeError(String::NewFromUtf8(isolate, "Last argument must be a number")));
        return;
    }


    try {
        const std::string url = to_string(args[0]);
        const std::string hash = to_string(args[1]);
        const std::time_t ttl = static_cast<std::time_t>(args[2]->ToNumber(isolate)->Value());

        const Vdirect* obj = ObjectWrap::Unwrap<Vdirect>(args.Holder());
        const bool valid = getValidator(obj->storage, ttl).valid(url, hash);

        args.GetReturnValue().Set(Boolean::New(isolate, valid));
    } catch (const std::exception& ex) {
        args.GetReturnValue().Set(Boolean::New(isolate, false));
    }
}

}
