#pragma once

#include <mail/webmail/clickhouse/include/format.h>
#include <mail/http_getter/client/include/module.h>
#include <yamail/data/deserialization/yaml.h>
#include <boost/fusion/include/adapt_struct.hpp>
#include <boost/range/algorithm/find_if.hpp>
#include <mail/webmail/corgi/include/types_error.h>
#include <yplatform/module_registration.h>
#include <mail/webmail/http_api_helpers/include/find_dependency.h>
#include <butil/http/headers.h>
#include <boost/algorithm/string.hpp>
#include <fstream>
#include <format>


BOOST_FUSION_DEFINE_STRUCT((clickhouse)(reflection), Query,
    (std::string, name)
    (std::string, query)
)

BOOST_FUSION_DEFINE_STRUCT((clickhouse)(reflection), QueryConf,
    (std::vector<clickhouse::reflection::Query>, config)
)

namespace clickhouse {

using namespace http_getter::detail::operators;
using namespace std::string_literals;

template<class Formatter, class Builder>
struct Request {
    template<typename Function>
    struct TransformerTraits {
        template<typename F, typename Result, typename... Args>
        static std::tuple<Args...> extract(Result (F::*)(Args...));

        template<typename F, typename Result, typename... Args>
        static std::tuple<Args...> extract(Result (F::*)(Args...) const);

        using Tuple = decltype(extract(&Function::operator()));

        static_assert(std::tuple_size_v<Tuple> == 1, "expect exactly one argument");

        using Argument = std::tuple_element_t<0, Tuple>;
        using Result = std::invoke_result_t<Function, Argument>;
    };

    inline static constexpr bool IsSelect = Builder::method == http_getter::Method::Get;
    inline static constexpr bool IsInsert = Builder::method == http_getter::Method::Post;

    static_assert(IsSelect || IsInsert, "expect GET or POST method");

    Request(std::string name, http_getter::TypedClientPtr client, Builder&& builder)
        : name(std::move(name))
        , client(std::move(client))
        , builder(std::move(builder))
    { }

    template<class Req>
    void makeRequest(Req&& req, OnResponse hook) {
        const auto answered = std::make_shared<bool>(false);
        client->req(std::move(req))->call(name, http_getter::withDefaultHttpWrap([answered, hook] (yhttp::response&& resp) {
            hook(mail_errors::error_code(), std::move(resp));
            *answered = true;
        }), [hook, answered] (mail_errors::error_code ec) {
            if (!*answered) {
                hook(make_error(corgi::DbError::permanentFail, ec.full_message()), yhttp::response{});
            }
        });
    }

    template<class Items>
    void perform(Items&& data, OnExecute hook) {
        static_assert(IsInsert, "insert method only");

        makeRequest(
            builder.body(Formatter::prepareInsertRequestData(std::move(data))),
            [hook] (mail_errors::error_code ec, yhttp::response) {
                hook(std::move(ec));
        });
    }

    template<class Transformer, class Hook>
    void perform(Transformer transformer, Hook hook) {
        static_assert(IsSelect, "select method only");
        using Reflection = typename TransformerTraits<Transformer>::Argument;
        using Result = typename TransformerTraits<Transformer>::Result;

        makeRequest(
            std::move(builder),
            Formatter::template wrapSelectRequestResponse<Transformer, Reflection, Result, Hook>(transformer, hook)
        );
    }

    std::string name;
    http_getter::TypedClientPtr client;
    Builder builder;
};

template<class Formatter = JSONEachRowFormatter>
struct Executer {
    template<class ... Queries>
    void initConfig(const yplatform::ptree& cfg, Queries&&... queries) {
        headers.add("X-ClickHouse-Database", cfg.get<std::string>("database"));
        headers.add("X-ClickHouse-Format",   std::string(Formatter::name()));
        headers.add("X-ClickHouse-User",     cfg.get<std::string>("user"));
        headers.add("X-ClickHouse-Key",      readPassword(cfg.get<std::string>("password_file")));


        loggerName = cfg.get<std::string>("dependencies.logger");
        getterModule = http_api::findDependency<http_getter::TypedClientModule>(cfg, "dependencies.http_getter");

        endpoint = yamail::data::deserialization::fromPtree<http_getter::TypedEndpoint>(cfg.get_child("endpoint"));

        loadFromFile(cfg.get<std::string>("queryconf"), queries...);
    }

    std::string readPassword(const std::string file) const {
        std::string password;
        std::ifstream in(file);
        if (!in) {
            throw std::runtime_error("cannot read file with password: "s + file);
        }
        std::getline(in, password);

        return password;
    }

    template<class Input, class ... Queries>
    void loadFrom(Input& in, Queries&&... queries) {
        reflection::QueryConf reflected;
        yamail::data::deserialization::fromYaml(in, reflected);

        static_assert(sizeof...(queries) > 0, "expected at least one query");

        boost::hana::for_each(boost::hana::make_tuple(queries...), [&] (const auto& q) {
            const auto name = q.c_str();
            const auto it = boost::find_if(reflected.config, [&] (auto&& el) { return el.name == name; });
            if (it == reflected.config.end()) {
                throw std::runtime_error(fmt::format("unexpected query: {}", name));
            } else if (nameToQuery.find(name) == nameToQuery.end()) {
                nameToQuery[name] = boost::trim_copy(it->query);
            } else {
                throw std::runtime_error(fmt::format("duplicated query: {}", name));
            }
        });
    }

    template<class ... Queries>
    void loadFromFile(const std::string& filename, Queries&&... queries) {
        std::ifstream in(filename);

        if (!in) {
            throw std::runtime_error(fmt::format("cannot read clickhouse queryconf file: {}", filename));
        }

        loadFrom(in, std::forward<Queries>(queries)...);
    }

    template<class Query, class ... Args>
    std::string formatQuery(Query query, Args&&... args) const {
        const std::string name = query.c_str();
        const auto it = nameToQuery.find(name);
        if (it == nameToQuery.end()) {
            throw std::runtime_error(fmt::format("unexpected query: {}", name));
        }
        return fmt::format(it->second, args...);
    }

    template<class Query, class Function, class ... Args>
    auto perform(const std::string& requestId, Query query, Function f, Args&&... args) const {
        http_getter::TypedClientPtr client = getterModule->create(
            requestId, http_getter::withLog(getterModule->httpLogger("", requestId))
        );

        auto builder = ((*client).*f)(endpoint)
                .getArgs("query"_arg=formatQuery(query, std::forward<Args>(args)...))
                .headers("headers"_hdr=headers);

        return Request<Formatter, decltype(builder)> {query.c_str(), client, std::move(builder)};
    }

    template<class Query, class ... Args>
    auto select(const std::string& requestId, Query query, Args&&... args) const {
        return perform(
            requestId, query,
            &http_getter::TypedClient::toGET,
            std::forward<Args>(args)...
        );
    }

    template<class Query, class ... Args>
    auto insert(const std::string& requestId, Query query, Args&&... args) const {
        return perform(
            requestId, query,
            &http_getter::TypedClient::toPOST,
            fmt::arg("format", "FORMAT "s + std::string(Formatter::name())),
            std::forward<Args>(args)...
        );
    }

    http::headers headers;

    std::string loggerName;
    http_getter::TypedModulePtr getterModule;

    http_getter::TypedEndpoint endpoint;

    std::map<std::string, std::string> nameToQuery;
};

}
