#pragma once

#include <yamail/data/deserialization/json_reader.h>
#include <yamail/data/serialization/json_writer.h>
#include <mail/webmail/corgi/include/types_error.h>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/find_if.hpp>
#include <boost/range/algorithm/remove_if.hpp>
#include <mail/webmail/clickhouse/include/types.h>


namespace clickhouse {

template<class Format, class Transformer, class Reflection, class Callback>
struct SequenceResponse {
    using Vec = std::vector<std::string>;

    std::shared_ptr<Vec> range;
    Callback cb;
    mutable typename Vec::const_iterator iter;
    mutable Transformer transformer;

    SequenceResponse(Vec r, Transformer t, Callback cb)
        : range(std::make_shared<Vec>(std::move(r)))
        , cb(cb)
        , iter(range->begin())
        , transformer(t)
    { }

    void operator()() const {
        if (iter != range->end()) {
            Reflection item;
            try {
                Format::deserialize(*iter++, item);

                cb(
                    mail_errors::error_code(),
                    io_result::hooks::detail::Cursor(transformer(item), *this)
                );
            } catch (const std::exception& ex) {
                cb(make_error(corgi::DbError::permanentFail, ex.what()));
            }
        } else {
            cb(mail_errors::error_code());
        }
    }
};

struct JsonFormat {
    template<class T>
    static std::string serialize(const T& val) {
        return yamail::data::serialization::JsonWriter(val).result();
    }

    template<class T>
    static void deserialize(const std::string& str, T& val) {
        yamail::data::deserialization::fromJson(str, val);
    }
};

template<class Format>
struct EachRow {
    static auto extractStrings(std::string body) {
        std::vector<std::string> strings;
        boost::trim(body);
        boost::split(strings, body, boost::is_any_of("\n"), boost::token_compress_on);
        strings.erase(
            boost::remove_if(strings, [] (auto&& s) { return s.empty(); }),
            strings.end()
        );

        return strings;
    }

    template<class Vec>
    static std::string prepareInsertRequestData(Vec&& data) {
        return boost::join(
            std::move(data)
            | boost::adaptors::transformed([] (auto&& v) {
                return Format::serialize(v);
            })
            , "\n"
        );
    }

    template<class Transformer, class Reflection, class Result, class Hook>
    static OnResponse wrapSelectRequestResponse(Transformer transform, Hook h) {
        return [transform, h] (mail_errors::error_code ec, yhttp::response resp) {
            if (ec) {
                h(std::move(ec));
            } else {
                SequenceResponse<Format, Transformer, Reflection, Hook> (
                    extractStrings(resp.body), transform, std::move(h)
                )();
            }
        };
    }
};

struct JSONEachRowFormatter: public EachRow<JsonFormat> {
    static std::string_view name() { return "JSONEachRow"; }
};

}
