#include <mail/webmail/commondb/include/settings.h>
#include <mail/webmail/commondb/include/logger.h>
#include <pgg/database/pool_factory.h>
#include <pgg/query/ids.h>
#include <pgg/query/boundaries.h>
#include <pgg/query/query_register.h>
#include <pgg/factory.h>
#include <yplatform/application/find.h>
#include <fstream>


namespace commondb {

struct WithExecuters {
    template<class QueriesRegister, class ParametersRegister>
    void initExecuters(const std::string& reactorName, const PgSettings& pgSettings,
                       QueriesRegister&& queries, ParametersRegister&& parameters) {
        using namespace std::string_literals;

        if (pgSettings.password_file) {
            std::ifstream in(*pgSettings.password_file);
            if (!in) {
                throw std::runtime_error("cannot read file with password: "s + *pgSettings.password_file);
            }
            std::getline(in, password);
        }

        queryConfPtr = pgg::readQueryConfFile(
            pgSettings.query_conf, std::forward<QueriesRegister>(queries),
            std::forward<ParametersRegister>(parameters)
        );

        reactor = yplatform::find_reactor<std::shared_ptr>(reactorName);

        for (std::size_t i = 0; i < reactor->size(); i++) {
            pools.push_back(pgg::ConnectionPoolFactory()
                .limit(pgSettings.max_total_pools_capacity)
                .connectTimeout(pgSettings.connect_timeout_ms)
                .queueTimeout(pgSettings.queue_timeout_ms)
                .queryTimeout(pgSettings.query_timeout_ms)
                .maxConnections(pgSettings.max_connections)
                .asyncResolve(pgSettings.async_resolve)
                .ipv6Only(pgSettings.ipv6_only)
                .dnsCacheTTL(pgSettings.dns_cache_ttl_sec)
                .ioService(*(*reactor)[i]->io())
                .product()
            );
        }

        readWriteConnectionString += pgSettings.connection_string + " target_session_attrs=read-write";
        anyConnectionString += pgSettings.connection_string + " target_session_attrs=any";
        logPa = pgSettings.log_pa;
        user = pgSettings.user;
    }

    template<class Logger>
    pgg::logging::LogPtr makePggLogger(Logger logger) const {
        return commondb::pggLogger(password, std::move(logger));
    }

    std::size_t passwordSize() const {
        return password.size();
    }

    std::shared_ptr<pgg::RequestExecutor> readWriteExecutor(const std::string& reqId, pgg::logging::LogPtr logger, pgg::profiling::LogPtr profiler) const {
        return createExecutor<pgg::MasterOnly>(readWriteConnectionString, reqId, std::move(logger), std::move(profiler));
    }

    std::shared_ptr<pgg::RequestExecutor> anyExecutor(const std::string& reqId, pgg::logging::LogPtr logger, pgg::profiling::LogPtr profiler) const {
        return createExecutor<pgg::ReadReplicaThenMaster>(anyConnectionString, reqId, std::move(logger), std::move(profiler));
    }

private:
    template<class Strategy>
    std::shared_ptr<pgg::RequestExecutor> createExecutor(const std::string& connStr, const std::string& reqId,
                                                         pgg::logging::LogPtr logger, pgg::profiling::LogPtr profiler) const {
        return std::make_shared<pgg::RequestExecutor>(
            pgg::Factory::product<Strategy>(
                "shard",
                pgg::createFakeShardResolverFactory(connStr, "shard"),
                params(reqId, std::move(logger), std::move(profiler))
            )
        );
    }

    pgg::ConnectionPoolPtr pool() const {
        return pools[reactor->get_index()];
    }

    pgg::FactoryParams params(const std::string& reqId, pgg::logging::LogPtr logger, pgg::profiling::LogPtr profiler) const {
        return pgg::FactoryParams {
            .connPool=pool(),
            .profiler=logPa ? std::move(profiler) : nullptr,
            .queryConf=queryConfPtr,
            .logger=std::move(logger),
            .credentials=pgg::Credentials{.user=user, .password=password},
            .requestInfo=pgg::RequestInfo{.requestId=reqId}
        };
    }

    pgg::QueryConf queryConfPtr;
    std::vector<pgg::ConnectionPoolPtr> pools;
    std::shared_ptr<yplatform::reactor> reactor;
    std::string password;
    std::string readWriteConnectionString;
    std::string anyConnectionString;
    bool logPa = false;
    std::string user;
};

}
