#pragma once

#include <string>
#include <set>
#include <boost/asio/spawn.hpp>
#include <yamail/expected.h>
#include <boost/serialization/strong_typedef.hpp>
#include <mail/webmail/http_api_helpers/include/error.h>


namespace corgi {

BOOST_STRONG_TYPEDEF(std::int64_t, Uid)
BOOST_STRONG_TYPEDEF(std::int64_t, OrgId)
BOOST_STRONG_TYPEDEF(std::int64_t, DepartmentId)
BOOST_STRONG_TYPEDEF(std::int64_t, GroupId)

inline Uid makeUid(const std::string& val) {
    return Uid(std::stoll(val));
}

inline OrgId makeOrgId(const std::string& val) {
    return OrgId(std::stoll(val));
}

inline DepartmentId makeDepartmentId(std::int64_t val) {
    return DepartmentId(val);
}

inline GroupId makeGroupId(std::int64_t val) {
    return GroupId(val);
}

using Uids = std::vector<Uid>;
using UidsSet = std::set<Uid>;
using RequestId = std::string;
using Departments = std::vector<DepartmentId>;
using DepartmentsSet = std::set<DepartmentId>;
using Groups = std::vector<GroupId>;
using GroupsSet = std::set<GroupId>;

struct ResolveUsers {
    std::optional<Departments> includeDepartments;
    std::optional<Departments> excludeDepartments;
    std::optional<Uids> includeUids;
    std::optional<Uids> excludeUids;
    std::optional<Groups> includeGroups;
    std::optional<Groups> excludeGroups;

    bool empty() const {
        return !(excludeDepartments || includeDepartments || excludeUids || includeUids || includeGroups || excludeGroups);
    }
};

enum class OrganizationState {
    active,
    disabled,
    frozen
};

struct Organization {
    OrganizationState state;
    std::time_t doomDate;
    std::time_t created;
};

struct CommonParams {
    Uid adminUid;
    OrgId orgId;
    RequestId requestId;

    auto operator<=>(const CommonParams&) const = default;
};

struct PageParams {
    using From = std::optional<std::string>;

    PageParams() = default;

    const From& first() const {
        return first_;
    }

    unsigned count() const {
        return count_;
    }

    void first(From val) {
        first_ = std::move(val);
    }

    void count(unsigned val) {
        if (!val) {
            throw mail_errors::system_error(make_error(http_api::Error::invalidArgument, "cannot be zero"));
        }
        count_ = val;
    }

private:
    From first_;
    unsigned count_;
};

struct StrongPageParams {
    std::int64_t first;
    unsigned count;

    StrongPageParams() = default;
    StrongPageParams(const PageParams& page, std::int64_t def)
        : first(page.first() ? std::stoll(*page.first()) : def)
        , count(page.count())
    { }
};

struct SettingParams {
    Uid uid;
    RequestId requestId;
};

struct OrganizationParams {
    OrgId orgId;
    RequestId requestId;

    auto operator<=>(const OrganizationParams&) const = default;
};

}
