#include <mail/webmail/corgi/include/resolve/blackbox.h>
#include <mail/webmail/corgi/include/trycatch.h>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <yamail/data/deserialization/json_reader.h>
#include <butil/network/idn.h>


namespace corgi {
namespace bb {
struct Uid {
    boost::optional<bool> hosted;
    boost::optional<std::string> value;
};

struct User {
    Uid uid;
};

struct BbException {
    std::string value;
    int id;
};

struct Response {
    boost::optional<std::vector<User>> users;
    boost::optional<BbException> exception;
};
}

static constexpr int DB_EXCEPTION = 10;

template<class Range>
yamail::expected<std::set<Uid>> pddUidsInternal(const Range& uidsToCheck,
                                                const http_getter::TypedClient& client,
                                                const http_getter::TypedEndpoint& bb,
                                                boost::asio::yield_context yield) {
    using namespace http_getter::detail::operators;

    const auto yy = io_result::make_yield_context(yield);

    yamail::expected<std::set<Uid>> result = make_unexpected(RemoteServiceError::blackbox, "unknown error");

    auto req = client.toGET(bb).getArgs(
        "userip"_arg="127.0.0.1", "method"_arg="userinfo",
        "emails"_arg="getdefault", "format"_arg="json",
        "uid"_arg=boost::algorithm::join(uidsToCheck | boost::adaptors::transformed([](const auto u) { return std::to_string(u); } ), ",")
    );

    client.req(std::move(req))->call("blackbox", [&](yhttp::response resp) {
        const unsigned status = resp.status;
        if (http_getter::helpers::successCode(status)) {
            http_getter::Result parseResult = http_getter::Result::fail;
            trycatch(result, [&] () {
                const auto response = yamail::data::deserialization::fromJson<bb::Response>(resp.body);

                if (response.exception) {
                    if (response.exception->id == DB_EXCEPTION) {
                        parseResult = http_getter::Result::retry;
                    } else {
                        result = make_unexpected(RemoteServiceError::blackbox, response.exception->value);
                    }
                } else if (!response.users || response.users->size() != boost::size(uidsToCheck)) {
                    result = make_unexpected(RemoteServiceError::blackbox, "strange number of users");
                } else {
                    std::set<Uid> uids;
                    for (const bb::User& u: *response.users) {
                        if (u.uid.hosted.get_value_or(false) && u.uid.value) {
                            uids.insert(makeUid(*u.uid.value));
                        }
                    }

                    result = uids;
                }
            });

            if (result) {
                parseResult = http_getter::Result::success;
            }
            return parseResult;
        } else {
            return http_getter::helpers::retriableCode(status) ? http_getter::Result::retry
                                                               : http_getter::Result::fail;
        }
    }, yy);

    return result;
}

yamail::expected<UidsSet> pddUids(const std::vector<Uid>& uids,
                                  const http_getter::TypedClient& client,
                                  const ResolverConfig& config,
                                  boost::asio::yield_context yield,
                                  const unsigned chunkSize) {
    std::set<Uid> result;
    std::size_t first = 0;
    while (first < uids.size()) {
        const auto range = boost::make_iterator_range_n(
            uids.begin() + first,
            first + chunkSize > uids.size() ? uids.size() - first : chunkSize
        );

        const auto pddUids = pddUidsInternal(range, client, config.blackbox, yield);
        if (!pddUids) {
            return pddUids;
        }

        result.insert(pddUids.value().begin(), pddUids.value().end());
        first += chunkSize;
    }
    return yamail::expected<std::set<Uid>>(result);
}

}

BOOST_FUSION_ADAPT_STRUCT(corgi::bb::Uid,
    hosted,
    value
)

BOOST_FUSION_ADAPT_STRUCT(corgi::bb::User,
    uid
)

BOOST_FUSION_ADAPT_STRUCT(corgi::bb::BbException,
    id,
    value
)

BOOST_FUSION_ADAPT_STRUCT(corgi::bb::Response,
    users,
    exception
)
