#include <mail/webmail/corgi/include/resolve/directory.h>
#include <mail/webmail/corgi/include/trycatch.h>
#include <yamail/data/deserialization/json_reader.h>
#include <iterator>


namespace corgi {
namespace dir {
struct Group {
    std::int64_t id;
};

struct UserFromDirectory {
    std::int64_t id;
    bool is_admin;
    std::optional<std::int64_t> department_id;
    std::vector<Group> groups;
};

struct Links {
    std::optional<std::string> next;
};

struct UsersResponse {
    Links links;
    std::vector<UserFromDirectory> result;
};

struct Parent {
    std::optional<std::int64_t> id;
};

struct Department {
    std::int64_t id;
    Parent parent;
};

struct DepartmentsResponse {
    Links links;
    std::vector<Department> result;
};
}

using namespace http_getter::detail::operators;

HttpArguments parseUrl(const std::string& url) {
    HttpArguments args;
    args.fromUrl(url);
    return args;
}

yamail::expected<bool> isAdmin(const CommonParams& common, const http_getter::TypedClient& client,
                               const ResolverConfig& config, boost::asio::yield_context yield) {

    yamail::expected<bool> result = make_unexpected(RemoteServiceError::directory, "unknown error");

    const auto handler = http_getter::withDefaultHttpWrap([&] (yhttp::response resp) {
        trycatch(result, [&] () {
            result = yamail::data::deserialization::fromJson<dir::UserFromDirectory>(resp.body).is_admin;
        });
        return result ? http_getter::Result::success : http_getter::Result::retry;
    });

    auto req = client
            .toGET(config.directorySingleUser.format(fmt::arg("uid", std::to_string(common.adminUid))))
            .getArgs("fields"_arg="is_admin,department_id,groups")
            .headers("X-Org-Id"_hdr=std::to_string(common.orgId))
    ;

    client.req(std::move(req))->call("is_admin", handler, io_result::make_yield_context(yield));

    return result;
}

template<class Result, class ParseAs, class Transformer, class TemplateReq>
yamail::expected<Result> requestWithNext(TemplateReq&& templateRequest, const std::string& requestName,
                                         Transformer transformer,
                                         const http_getter::TypedClient& client, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    yamail::expected<ParseAs> result = make_unexpected(corgi::RemoteServiceError::directory, "unknown error");

    const auto handler = http_getter::withDefaultHttpWrap([&] (yhttp::response resp) {
        trycatch(result, [&] () {
            result = yamail::data::deserialization::fromJson<ParseAs>(resp.body);
        });
        return result ? http_getter::Result::success : http_getter::Result::retry;
    });

    std::string next;
    Result ret;

    do {
        auto req = templateRequest;

        client
            .req(req.getArgs("args"_arg=parseUrl(next)))
            ->call(requestName, handler, yy);

        if (!result) {
            return yamail::make_unexpected(result.error());
        }

        boost::copy(
            result.value().result
            | boost::adaptors::transformed(transformer)
            , std::inserter(ret, ret.end())
        );
        next = result.value().links.next.value_or("");
    } while (!next.empty());

    return ret;
}

yamail::expected<UidToDirectoryUserMap> resolveOrgId(OrgId orgId,
                                                     const http_getter::TypedClient& client,
                                                     const ResolverConfig& config,
                                                     boost::asio::yield_context yield) {

    const auto templateRequest = client
            .toGET(config.directoryUsers)
            .getArgs("fields"_arg="is_admin,department_id,groups")
            .headers("X-Org-Id"_hdr=std::to_string(orgId))
    ;

    const auto transform = [] (const auto& u) {
        Uid uid(u.id);
        DirectoryUser ret {.uid=uid, .admin=u.is_admin};

        if (u.department_id.has_value()) {
            ret.departmentId = makeDepartmentId(*u.department_id);
        }

        std::transform(
            u.groups.begin(), u.groups.end(), std::inserter(ret.groups, ret.groups.begin()),
            [] (const auto& rawGroup) { return makeGroupId(rawGroup.id); }
        );

        return std::make_pair(uid, std::move(ret));
    };

    return requestWithNext<UidToDirectoryUserMap, dir::UsersResponse>(
        templateRequest, "resolve_organization", transform, client, yield
    );
}

yamail::expected<DepartmentsTree> resolveDepartmentsTree(OrgId orgId,
                                                         const http_getter::TypedClient& client,
                                                         const ResolverConfig& config,
                                                         boost::asio::yield_context yield) {

    const auto templateRequest = client
            .toGET(config.directoryDepartments)
            .getArgs("fields"_arg="parent")
            .headers("X-Org-Id"_hdr=std::to_string(orgId))
    ;

    const auto deps = requestWithNext<std::vector<dir::Department>, dir::DepartmentsResponse>(
        templateRequest, "departments_tree", [] (auto&& dep) { return dep; }, client, yield
    );

    if (!deps) {
        return deps.get_unexpected();
    }

    DepartmentsTree result;
    for (const auto& dep: *deps) {
        if (dep.parent.id) {
            result[DepartmentId(*dep.parent.id)].push_back(DepartmentId(dep.id));
        } else {
            result[DepartmentId(dep.id)] = Departments();
        }
    }

    return result;
}

}

BOOST_FUSION_ADAPT_STRUCT(corgi::dir::Parent, id)

BOOST_FUSION_ADAPT_STRUCT(corgi::dir::Department, id, parent)

BOOST_FUSION_ADAPT_STRUCT(corgi::dir::DepartmentsResponse, links, result)

BOOST_FUSION_ADAPT_STRUCT(corgi::dir::Group, id)

BOOST_FUSION_ADAPT_STRUCT(corgi::dir::UserFromDirectory, id, is_admin, department_id, groups)

BOOST_FUSION_ADAPT_STRUCT(corgi::dir::Links, next)

BOOST_FUSION_ADAPT_STRUCT(corgi::dir::UsersResponse, links, result)
