#include <mail/webmail/corgi/include/resolve/include_exclude.h>
#include <mail/webmail/corgi/include/types_error.h>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>


namespace corgi {
namespace detail {
bool isAnyOfSetsContainUser(const UidsSet& uids, const DepartmentsSet& deps, const GroupsSet& groups, const DirectoryUser& user) {
    if (uids.contains(user.uid)) {
        return true;
    }

    if (user.departmentId && deps.contains(*user.departmentId)) {
        return true;
    }

    GroupsSet tmp;
    std::set_intersection(
        user.groups.begin(), user.groups.end(),
        groups.begin(), groups.end(),
        std::inserter(tmp, tmp.end())
    );

    return !tmp.empty();
}

bool includeUsers(const UidsSet& uids, const DepartmentsSet& deps, const GroupsSet& groups, const DirectoryUser& user) {
    if (uids.empty() && deps.empty() && groups.empty()) {
        return true;
    }

    return isAnyOfSetsContainUser(uids, deps, groups, user);
}

bool excludeUsers(const UidsSet& uids, const DepartmentsSet& deps, const GroupsSet& groups, const DirectoryUser& user) {
    return !isAnyOfSetsContainUser(uids, deps, groups, user);
}

template<class Set>
void checkIntersection(const Set& a, const Set& b, std::string msg) {
    Set tmp;
    std::set_intersection(
        a.begin(), a.end(), b.begin(), b.end(),
        std::inserter(tmp, tmp.end())
    );

    if (!tmp.empty()) {
        throw mail_errors::system_error(make_error(ResolveOrganizationError::includeExcludeIntersection, std::move(msg)));
    }
}
template<class Vector, class Set>
void insert(const std::optional<Vector>& from, Set& to) {
    if (from) {
        boost::copy(*from, std::inserter(to, to.end()));
    }
}

DepartmentsSet resolveSubDepartments(DepartmentsSet toResolve, const DepartmentsTree& tree) {
    DepartmentsSet result = toResolve;
    while(!toResolve.empty()) {
        DepartmentsSet newbie;

        for (const DepartmentId& id: toResolve) {
            if (tree.contains(id)) {
                boost::copy(tree.at(id), std::inserter(newbie, newbie.end()));
            }
        }

        boost::copy(newbie, std::inserter(result, result.end()));
        toResolve.swap(newbie);
    }
    return result;
}
}

IncludeAndExcludeByUidAndDepartment::IncludeAndExcludeByUidAndDepartment(const ResolveUsers& resolve, const DepartmentsTree& tree) {
    detail::insert(resolve.excludeDepartments, excludeDepartments);
    detail::insert(resolve.includeDepartments, includeDepartments);
    detail::insert(resolve.excludeUids, excludeUids);
    detail::insert(resolve.includeUids, includeUids);
    detail::insert(resolve.excludeGroups, excludeGroups);
    detail::insert(resolve.includeGroups, includeGroups);

    includeDepartments = detail::resolveSubDepartments(includeDepartments, tree);
    excludeDepartments = detail::resolveSubDepartments(excludeDepartments, tree);

    detail::checkIntersection(includeDepartments, excludeDepartments, "include and exclude departments are intersected");
    detail::checkIntersection(includeUids, excludeUids, "include and exclude uids are intersected");
    detail::checkIntersection(includeGroups, excludeGroups, "include and exclude groups are intersected");
}

Uids IncludeAndExcludeByUidAndDepartment::filter(const DirectoryUsers& org) const {
    Uids filtered;
    boost::copy(
        org
        | boost::adaptors::filtered([this] (auto&& u) { return detail::includeUsers(includeUids, includeDepartments, includeGroups, u); })
        | boost::adaptors::filtered([this] (auto&& u) { return detail::excludeUsers(excludeUids, excludeDepartments, excludeGroups, u); })
        | boost::adaptors::transformed([]  (auto&& u) { return u.uid; })
        , std::back_inserter(filtered)
    );

    return filtered;
}

}
