from math import ceil, floor
from .types import (
    Hostname,
    Alert,
    Escalation,
)

MAX_WIDTH = 5
MAX_LIMIT = 20


def make_widget(env, widget, position_gen, suffix, limit):
    filters = []
    for namespace in widget.namespaces or ['']:
        for host in widget.hosts or [Hostname.any()]:
            for service in widget.services or ['']:
                alert = {}
                if namespace:
                    alert['namespace'] = namespace
                if service:
                    alert['service'] = service
                if host:
                    alert['host'] = host(env)
                if alert:
                    filters.append(alert)
    if isinstance(widget, Alert):
        return make_check_widget(widget, filters, position_gen, suffix, limit)
    elif isinstance(widget, Escalation):
        return make_escalation_widget(widget, filters, suffix)
    return None


def make_check_widget(alert, filters, position_gen, suffix, limit):
    pos = next(position_gen)
    options = {
        'filters': filters,
        'statuses': ['CRIT', 'WARN'],
        'limit': limit or MAX_LIMIT,
        'include_mutes': False,
        'sort': {'field': 'DEFAULT', 'order': 'DESC'},
    }
    return {
        'name': alert.name + suffix,
        'aggregate_checks_options': options,
        'component_type': 'AGGREGATE_CHECKS',
        'view_type': 'COMPACT',
        'row': pos.row,
        'col': pos.col,
    }


def make_escalation_widget(escalation, filters, suffix):
    options = {
        'filters': filters,
        'only_running': False,
        'page_size': 20,
        'page': 0,
    }
    return {
        'name': escalation.name + suffix,
        'escalations_options': options,
        'component_type': 'ESCALATIONS',
        'view_type': 'DETAILED',
    }


def make_dashboard(env, dashboard, owners):
    envs = env.split('_')
    checks = []
    pos_gen = position_gen(len(dashboard.widgets), dashboard.width)
    for e in envs:
        pos_gen.newline()
        suffix = '_' + e if len(envs) > 1 else ''
        checks += [make_widget(e, w, pos_gen, suffix, dashboard.limit) for w in dashboard.widgets]
    return {
        'name': dashboard.name.format(env),
        'address': dashboard.address.format(env),
        'owners': owners,
        'components': checks,
    }


def make_env(env, owners):
    return [make_dashboard(env.name, d, owners) for d in env.dashboards]


class Position:
    def __init__(self, width):
        self.__width = width
        self.__number = None

    def __inc(self):
        if self.__number is None:
            self.__number = 0
        else:
            self.__number += 1

    def newline(self):
        if self.__number is not None:
            self.__number = (ceil(self.__number / self.__width) + 1) * self.__width - 1

    @property
    def row(self):
        return floor(self.__number / self.__width) + 1

    @property
    def col(self):
        return (self.__number % self.__width) + 1

    def __next__(self):
        self.__inc()
        return self


def position_gen(total, width):
    return Position(width or (ceil(total / ceil(total / MAX_WIDTH))))
