import requests

from .helpers import (
    get_oauth_from_yav,
    get_owners,
    regenerate_dashboard,
)
from .types import (
    Hostname,
    Alert,
    Escalation,
    Dashboard,
    Environment,
    merge_alerts,
)
from .gens import make_env


# -------- Alerts --------
DB_REPLICATION_LAG = [
    Alert('replication lag', ['mail.maildev.dashboard_only'], [Hostname.full('mail.maildb.dbaas.{}*')],
          ['replication_lag', 'total_replication_lag']),
]

DB_ALERTS = [
    Alert('trend replication lag', ['mail.maildev'], [Hostname.full('mail.maildb.dbaas.{}*')],
          ['trend_replication_lag']),
    Alert('sync rep wait events', ['mail.maildev'], [Hostname.full('mail.maildb.dbaas.{}*')],
          ['sync_rep_wait_events']),
    Alert('cpu usage percent', ['mail.maildev'], [Hostname.full('mail.maildb.dbaas.{}*')],
          ['cpu_usage_percent']),
    Alert('used space', ['mail.maildev'], [Hostname.full('mail.maildb.dbaas.{}*')],
          ['max_used_space']),
    Alert('premade partitions', ['mail.maildev'], [Hostname.full('mail.maildb.dbstats.{}*')],
          ['premade_partitions.*']),
    Alert('query time 90 perc', ['mail.maildev'], [Hostname.full('mail.maildb.dbaas.{}*')],
          ['query_90_percentile_ms']),
    Alert('pop3 init queue', ['mail.maildev'], [Hostname.full('mail.maildb.dbstats.{}*')],
          ['pop3_init_queue']),
]

SHIVA_ALERTS = [
    Alert('stuck tasks', ['mail.maildev'], [Hostname.short('mail.shiva.{}')], ['stuck_task.*']),
    Alert('tasks fails', ['mail.maildev'], [Hostname.short('mail.shiva.{}')], ['task_fail_perc.*']),
]

HUSKY_ALERTS = [
    Alert('husky-api 5xx', ['mail.maildev'], [Hostname.short('mail.husky.{}')], ['husky_api.5xx']),
    Alert('clone_user', ['mail.maildev'], [Hostname.short('mail.husky.{}')], ['husky.db_task.clone_user.*']),
    Alert('delete_mail_user', ['mail.maildev'], [Hostname.short('mail.husky.{}')], ['husky.db_task.delete_mail_user.*']),
    Alert('delete_shards_user', ['mail.maildev'], [Hostname.short('mail.husky.{}')], ['husky.db_task.delete_shards_user.*']),
]

CMB_ALERTS = [
    Alert('api errors', ['mail.callmeback'], [Hostname.short('mail.callmeback.{}')], ['api-errors']),
    Alert('api timings', ['mail.callmeback'], [Hostname.short('mail.callmeback.{}')], ['api-timings']),
    Alert('lost_buckets', ['mail.callmeback'], [Hostname.short('mail.callmeback.{}')], ['lost_buckets']),
]

DOBBY_ALERTS = [
    Alert('dobby queue', ['mail.doberman'], [Hostname.full('mail.maildb.dbstats.{}*')],
          ['doberman.change_queue_size']),
    Alert('dobby failed subscriptions', ['mail.doberman'], [Hostname.full('mail.maildb.dbstats.{}*')],
          ['doberman.failed_subscriptions']),
    Alert('dobby dead workers', ['mail.doberman'], [Hostname.full('mail.maildb.dbstats.{}*')],
          ['doberman.dead_workers']),
    Alert('dobby workerless subscriptions', ['mail.doberman'], [Hostname.full('mail.maildb.dbstats.{}*')],
          ['doberman.workerless_subscriptions']),
]

YORK_HOSTS = {
    'prod': 'mail.york.production',
    'corp': 'mail.york-corp.production',
    'test': 'mail.york-corp.testing',
}

YORK_ALERTS = [
    Alert('york 5xx', ['mail.york'], [Hostname.by_dict(YORK_HOSTS)], ['york.5xx']),
    Alert('york errors', ['mail.york'], [Hostname.by_dict(YORK_HOSTS)], ['york.york.errors']),
    Alert('york worker errors', ['mail.york'], [Hostname.by_dict(YORK_HOSTS)], ['york.worker.errors']),
]


# -------- Escalations --------
ALL_CALLS = [
    Escalation('escalations', ['mail.maildev', 'mail.callmeback', 'mail.doberman', 'mail.york'], [], []),
]


# -------- Dashboards --------
DB_DASHBOARD = Dashboard('db alerts {}', 'mailpg_db_alerts_{}', DB_REPLICATION_LAG + DB_ALERTS)
SHIVA_DASHBOARD = Dashboard('shiva alerts {}', 'mailpg_shiva_alerts_{}', SHIVA_ALERTS)
HUSKY_DASHBOARD = Dashboard('husky alerts {}', 'mailpg_husky_alerts_{}', HUSKY_ALERTS)
CMB_DASHBOARD = Dashboard('callmeback alerts {}', 'mailpg_callmeback_alerts_{}', CMB_ALERTS)
SHARED_FOLDERS_DASHBOARD = Dashboard('shared folders alerts {}', 'mailpg_shared_folders_alerts_{}', DOBBY_ALERTS + YORK_ALERTS)

CALLS_DASHBOARD = Dashboard('escalations', 'mailpg_escalations', ALL_CALLS)

ZOMB_DASHBOARD = Dashboard('dba dashboard {}', 'mailpg_dba_alerts_{}',
                           DB_ALERTS +
                           [merge_alerts('husky', HUSKY_ALERTS)] +
                           [merge_alerts('shiva', SHIVA_ALERTS)] +
                           [merge_alerts('doberman', DOBBY_ALERTS)] +
                           [merge_alerts('york', YORK_ALERTS)],
                           width=6, limit=2)

# -------- Environments --------
PROD_ENV = Environment('prod', [DB_DASHBOARD, SHIVA_DASHBOARD, HUSKY_DASHBOARD, CMB_DASHBOARD, CALLS_DASHBOARD])
CORP_ENV = Environment('corp', [DB_DASHBOARD, SHIVA_DASHBOARD, HUSKY_DASHBOARD, SHARED_FOLDERS_DASHBOARD])
TEST_ENV = Environment('test', [DB_DASHBOARD, SHIVA_DASHBOARD, HUSKY_DASHBOARD, CMB_DASHBOARD])

ZOMB_ENV = Environment('prod_corp', [ZOMB_DASHBOARD])

GENERATE_ENVS = [
    PROD_ENV,
    CORP_ENV,
    TEST_ENV,
    ZOMB_ENV,
]


def main():
    juggler_token, abc_token = get_oauth_from_yav()
    owners = get_owners(abc_token)
    for env in GENERATE_ENVS:
        print('==== Will generate {} env ===='.format(env.name))
        for db in make_env(env, owners):
            print('======== Generate {} ========'.format(db['address']))
            resp = regenerate_dashboard(juggler_token, db)
            if resp.status_code == requests.codes.ok:
                print('======== Success for {} ========'.format(db['address']))
            else:
                print('======== Error for {} ========'.format(db['address']))
                print(resp.text)
                exit(1)


if __name__ == "__main__":
    main()
