import sys
import json
import yaml
import asyncio
import uvloop
from aiohttp import web, ClientSession
from pathlib import Path
from mail.python.theatre.app.settings import LogSettings
from mail.python.theatre.app.log_helpers.app_logs import init_app_logging

from .duty import on_duty
from .assigner import assign
from .meters import (
    TimeMeter,
    stat,
    set_custom_meter,
    set_primary_meter,
    set_fallback_meter,
)

client_session = None


def init_logging(config):
    log_config = config['log']
    log = LogSettings()
    log.dir_path = Path(log_config['dir'])
    log.level = log_config['level']
    init_app_logging(settings=log, env=None)


def get_session(loop):
    global client_session
    if client_session is None:
        client_session = ClientSession(loop=loop)
    return client_session


async def create_app(loop):
    with open(sys.argv[1]) as c:
        conf = yaml.full_load(c)

    init_logging(conf)

    with open(conf['tokens_file']) as t:
        tokens = json.load(t)

    meter = TimeMeter('fake_caller', conf['meter_delay'], conf['meter_ttl'])

    app = web.Application(loop=loop)
    app.router.add_get('/ping', lambda _: web.Response(text='pong\n'))
    app.router.add_get('/assign', lambda r: assign(get_session(loop), tokens, r))
    app.router.add_get('/on_duty', lambda _: on_duty(get_session(loop), tokens))
    app.router.add_get('/stat', lambda _: stat(meter))

    app.router.add_post('/call', lambda r: set_custom_meter(get_session(loop), tokens, r, meter))
    app.router.add_post('/call_primary', lambda r: set_primary_meter(get_session(loop), tokens, r, meter))
    app.router.add_post('/call_fallback', lambda r: set_fallback_meter(get_session(loop), tokens, r, meter))
    return app


def main():
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())
    web.run_app(
        create_app(asyncio.get_event_loop()), host='::1', port=8081)


if __name__ == '__main__':
    main()
