import json
import time
import copy
from aiohttp import web

from .duty import get_duty

CHECK_TEMPLATE = {
    'namespace': 'mailpg-duty',
    'host': 'yasm_fake_caller_alert',
    'service': 'fake_caller_alert',
    'refresh_time': 5,
    'ttl': 900,
    'aggregator': 'logic_or',
    'aggregator_kwargs': {'nodata_mode': 'force_ok'},
    'flaps': {
        'stable_time': 1,
        'critical_time': 1,
        'boost_time': 0
    },
    'tags': [
        'a_mark_yasm_fake_caller_alert',
        'a_itype_qloud',
        'a_prj_mail.mailpg.groupie',
    ],
    'children': [
        {
            'host': 'yasm_alert',
            'service': 'fake_caller_alert',
            'type': 'HOST'
        }
    ]
}


class TimeMeter(object):
    def __init__(self, name, delay, duration):
        self.__name = name
        self.__start = 0
        self.__delay = delay
        self.__duration = duration
        self.__session = None
        self.__tokens = None

    async def __value(self):
        if (time.time() - self.__start) < self.__delay:
            return 0
        elif (time.time() - self.__start) < (self.__delay + self.__duration):
            return 1
        else:
            await self.__reset_check()
            return 0

    async def __reset_check(self):
        if self.__session and self.__tokens:
            await update_check(self.__session, self.__tokens, '', '')
            self.__session = None
            self.__tokens = None

    async def to_metric(self):
        return [self.__name + '_ammm', await self.__value()]

    def reset(self, session, tokens):
        self.__session = session
        self.__tokens = tokens
        self.__start = time.time()

    def name(self):
        return self.__name


async def stat(meter: TimeMeter):
    meters = [await meter.to_metric()]
    return web.Response(text=json.dumps(meters))


async def set_meter(session, tokens, req, login, meter: TimeMeter):
    text = await req.text()
    await update_check(session, tokens, login, text)
    meter.reset(session, tokens)
    return web.Response(text='Will call {} in a minute'.format(login))


async def set_custom_meter(session, tokens, req, meter: TimeMeter):
    login = req.query.get('login', '')
    if not login:
        return web.Response(status=400, text='login required')
    return await set_meter(session, tokens, req, login, meter)


async def set_duty_meter(session, tokens, req, duty_type, meter: TimeMeter):
    duty = await get_duty(session, tokens)
    login = duty.get(duty_type)
    if not login:
        return web.Response(status=500, text='no {} duty'.format(duty_type))
    return await set_meter(session, tokens, req, login, meter)


async def set_primary_meter(session, tokens, req, meter: TimeMeter):
    return await set_duty_meter(session, tokens, req, 'primary', meter)


async def set_fallback_meter(session, tokens, req, meter: TimeMeter):
    return await set_duty_meter(session, tokens, req, 'fallback', meter)


async def update_check(session, tokens, login, message):
    new_check = copy.deepcopy(CHECK_TEMPLATE)
    new_check['pronounce'] = message[:200]
    if login:
        new_check['notifications'] = [{
            'template_name': 'phone_escalation',
            'template_kwargs': {
                'delay': 1,
                'logins': [login]
            }
        }]
    else:
        new_check['notifications'] = []
    headers = {
        'Authorization': 'OAuth {}'.format(tokens['juggler.oauth.token']),
        'Content-Type': 'application/json',
    }
    await session.post('https://juggler-api.yandex-team.ru/api/checks/add_or_update?do=1',
                       headers=headers, data=json.dumps(new_check))
