#pragma once

#include <optional>

#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>
#include <mail/webmail/http_api_helpers/include/error.h>

namespace http_api {

struct Context {
    ymod_webserver::request_ptr req;

    std::optional<std::string> optionalArg(const std::string& name) const {
        if (auto iter = req->url.params.find(name); iter != req->url.params.end()) {
            return std::make_optional(iter->second);
        } else {
            return std::nullopt;
        }
    }

    std::string arg(const std::string& name) const {
        if (auto iter = req->url.params.find(name); iter != req->url.params.end()) {
            return iter->second;
        }

        throw mail_errors::system_error(
            make_error(Error::missingParam, fmt::format("missing '{}'", name))
        );
    }

    std::optional<std::vector<std::string>> optionalArgs(const std::string& name) const {
        std::vector<std::string> ret;
        boost::copy(req->url.params.equal_range(name)
                    | boost::adaptors::map_values
                    , std::back_inserter(ret));

        if (ret.empty()) {
            return std::nullopt;
        }

        return ret;
    }

    std::vector<std::string> args(const std::string& name) const {
        if (auto opt = optionalArgs(name)) {
            return *opt;
        } else {
            throw mail_errors::system_error(
                make_error(Error::missingParam, fmt::format("missing '{}'", name))
            );
        }
    }

    std::optional<std::string> optionalHeader(std::string name) const {
        boost::algorithm::to_lower(name);
        if (auto iter = req->headers.find(name); iter != req->headers.end()) {
            return std::make_optional(iter->second);
        } else {
            return std::nullopt;
        }
    }

    std::string body() const {
        return std::string(req->raw_body.begin(), req->raw_body.end());
    }

    static void throwMissingParamException(const std::string& what) {
        throw mail_errors::system_error(make_error(Error::missingParam, what));
    }
};

}
