#pragma once

#include <yamail/expected.h>

#include <boost/asio/spawn.hpp>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <mail/tvm_guard/include/tvm_guard/module.h>

#include <mail/webmail/http_api_helpers/include/error.h>
#include <mail/webmail/http_api_helpers/include/response.h>
#include <mail/webmail/http_api_helpers/include/merge_args_as_tskv.h>
#include <mail/webmail/http_api_helpers/include/find_dependency.h>

namespace http_api {

template<class Result>
using WrappedFunction = std::function<yamail::expected<Result>(ymod_webserver::response_ptr, boost::asio::yield_context)>;

namespace detail {
template<class Function>
auto spawn(const boost::coroutines::attributes& attrs, yplatform::reactor& reactor, Function f) {
    return [attrs, f, &reactor] (ymod_webserver::response_ptr s) {
        boost::asio::spawn(*reactor.io(), [s, f] (boost::asio::yield_context yield) {
            f(s, yield);
        }, attrs);
    };
}

template<class T, class MakeResponseFn>
auto response(MakeResponseFn makeResponse, const WrappedFunction<T>& f) {
    return [=] (ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
        const auto response = makeResponse(s);
        try {
            if (auto resp = f(s, yield); resp) {
                response.with(std::move(resp).value());
            } else {
                response.with(std::move(resp).error());
            }
        } catch (const boost::system::system_error& ex) {
            response.with(mail_errors::error_code(ex.code(), ex.what()));
        } catch (const std::exception& ex) {
            response.with(make_error(Error::unexpectedException, ex.what()));
        } catch (const boost::coroutines::detail::forced_unwind&) {
            throw;
        } catch(...) {
            response.with(make_error(Error::unexpectedException, "strange type of exception"));
        }
    };
}

template<class T>
auto methodOnly(ymod_webserver::methods::http_method method, const WrappedFunction<T>& f) {
    return [method, f] (ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
        if (s->request()->method == method) {
            return f(s, yield);
        } else {
            std::string expected;
            switch(method) {
                case ymod_webserver::methods::mth_get: expected = "GET"; break;
                case ymod_webserver::methods::mth_post: expected = "POST"; break;
                default: expected = std::to_string(static_cast<int>(method)); break;
            }

            return yamail::make_expected_from_error<T>(
                make_error(Error::wrongHttpMethod, fmt::format("expected '{}' method", expected))
            );
        }
    };
}

template<class T, class Function>
WrappedFunction<T> methodGET(const Function& f) {
    return methodOnly(ymod_webserver::methods::mth_get, WrappedFunction<T>(f));
}

template<class T, class Function>
WrappedFunction<T> methodPOST(const Function& f) {
    return methodOnly(ymod_webserver::methods::mth_post, WrappedFunction<T>(f));
}

template<class T, class Function>
WrappedFunction<T> withLogging(LogArgsAsTskvConfig logArgsCfg, const Function& f) {
    return [logArgsCfg, f] (ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
        logArgsAsTskv(s, logArgsCfg);
        return f(std::move(s), std::move(yield));
    };
}

template<class Result, class MakeResponseFn>
void bindFn(std::shared_ptr<tvm_guard::Module> guarded,
            std::shared_ptr<ymod_webserver::server> server,
            const std::string& path, const boost::coroutines::attributes& attrs,
            MakeResponseFn makeResponse, std::shared_ptr<yplatform::reactor> reactor,
            WrappedFunction<Result> f) {
    guarded->bind(*server, "", { path }, [=] (ymod_webserver::response_ptr s) {
        spawn(attrs, *reactor, response<Result>(makeResponse, f)) (s);
    });
}

template<class Response>
auto responseCreator() {
    return [] (ymod_webserver::response_ptr s) {
        return Response(std::move(s));
    };
}
}

template<class Response>
struct BindInfo {
    std::shared_ptr<tvm_guard::Module> guarded;
    std::shared_ptr<ymod_webserver::server> server;
    boost::coroutines::attributes attributes;
    std::shared_ptr<yplatform::reactor> reactor;
};

template<class Result, class MakeResponseFn>
void bindGET(std::shared_ptr<tvm_guard::Module> guarded,
             std::shared_ptr<ymod_webserver::server> server,
             const std::string& path, const boost::coroutines::attributes& attrs,
             MakeResponseFn makeResponse, std::shared_ptr<yplatform::reactor> reactor,
             WrappedFunction<Result> f) {
    using namespace detail;
    bindFn(guarded, server, path, attrs, makeResponse, reactor, methodGET<Result>(f));
}

template<class Result, class MakeResponseFn>
void bindPOST(std::shared_ptr<tvm_guard::Module> guarded,
              std::shared_ptr<ymod_webserver::server> server,
              const std::string& path, const boost::coroutines::attributes& attrs,
              MakeResponseFn makeResponse, std::shared_ptr<yplatform::reactor> reactor,
              WrappedFunction<Result> f) {
    using namespace detail;
    bindFn(guarded, server, path, attrs, makeResponse, reactor, methodPOST<Result>(f));
}

template<class Result, class MakeResponseFn>
void bindPOST(std::shared_ptr<tvm_guard::Module> guarded,
              std::shared_ptr<ymod_webserver::server> server,
              const std::string& path, const boost::coroutines::attributes& attrs,
              MakeResponseFn makeResponse, std::shared_ptr<yplatform::reactor> reactor,
              const LogArgsAsTskvConfig& logArgsCfg,
              WrappedFunction<Result> f) {
    using namespace detail;
    bindFn(guarded, server, path, attrs,
           makeResponse, reactor, withLogging<Result>(logArgsCfg, methodPOST<Result>(f)));
}

template<class Result, class Response>
void bindGET(BindInfo<Response> info, const std::string& path, WrappedFunction<Result> f) {
    using namespace detail;
    bindFn(info.guarded, info.server, path, info.attributes, responseCreator<Response>(), info.reactor, methodGET<Result>(f));
}

template<class Result, class Response>
void bindPOST(BindInfo<Response> info, const std::string& path, WrappedFunction<Result> f) {
    using namespace detail;
    bindFn(info.guarded, info.server, path, info.attributes, responseCreator<Response>(), info.reactor, methodPOST<Result>(f));
}

template<class Result, class Response>
void bindPOST(BindInfo<Response> info, const LogArgsAsTskvConfig& logArgsCfg, const std::string& path, WrappedFunction<Result> f) {
    using namespace detail;
    bindFn(info.guarded, info.server, path, info.attributes, responseCreator<Response>(),
           info.reactor, withLogging<Result>(logArgsCfg, methodPOST<Result>(f)));
}

}
