#pragma once

#include <ymod_webserver/response.h>
#include <mail_errors/error_code.h>
#include <yamail/expected.h>
#include <yamail/data/serialization/json_writer.h>
#include <mail/webmail/http_api_helpers/include/error.h>
#include <mail/webmail/http_api_helpers/include/result.h>


namespace http_api {

class Response {
protected:
    static const std::string fatalJsonResponse;

    template<class T>
    void defaultJsonResponse(T&& resp) const {
        std::string body;
        try {
            body = yamail::data::serialization::toJson(resp).str();
        } catch (const std::exception& ex) {
            errorJsonResponse(ymod_webserver::codes::internal_server_error,
                              make_error(Error::unexpectedException, ex.what()));
            return;
        }

        try {
            logSuccessResponse(body);
        } catch (const std::exception& ex) {
            logExceptionToDefaultLogger(ex);
        }

        stream->set_code(ymod_webserver::codes::ok);
        stream->set_content_type("application", "json");
        stream->result_body(body);
    }

    void errorJsonResponse(ymod_webserver::codes::code code,
                           const mail_errors::error_code& ec) const;
    void logExceptionToDefaultLogger(const std::exception& ex) const;

    virtual void logErrorResponse(const std::string& body) const = 0;
    virtual void logSuccessResponse(const std::string& body) const = 0;
    virtual bool errorCodeToResponseCode(const mail_errors::error_code& ec, ymod_webserver::codes::code& code) const = 0;

    ymod_webserver::response_ptr stream;

public:

    Response(ymod_webserver::response_ptr s)
        : stream(std::move(s))
    { }

    virtual ~Response() { }

    void with(mail_errors::error_code&& ec) const;
    void with(PingResult&& resp) const;
    void with(EmptyResult&& resp) const;
};

}
