#include <mail/webmail/http_api_helpers/include/response.h>
#include <mail/webmail/http_api_helpers/include/error.h>
#include <mail/mail_errors/error_result/error_result.h>
#include <logdog/backend/yplatform_log.h>
#include <logdog/format/tskv.h>
#include <logdog/attributes/mail_attributes.h>

namespace http_api {

const auto defaultLogger = logdog::make_log(
    logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-http_api-tskv-log")),
    std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE)
);

const std::string Response::fatalJsonResponse = mail_errors::makeErrorResult(
    mail_errors::ErrorResult("http_api", "default error response", "unknown error")
);

void Response::errorJsonResponse(ymod_webserver::codes::code code,
                                 const mail_errors::error_code& ec) const {
    std::string body;
    try {
        body = mail_errors::makeErrorResult(ec);
    } catch (const std::exception&) {
        body = fatalJsonResponse;
        code = ymod_webserver::codes::internal_server_error;
    }

    try {
        logErrorResponse(body);
    } catch (const std::exception& ex) {
        logExceptionToDefaultLogger(ex);
    }

    stream->set_code(code);
    stream->set_content_type("application", "json");
    stream->result_body(body);
}

void Response::logExceptionToDefaultLogger(const std::exception& ex) const {
    LOGDOG_(defaultLogger, error,
            logdog::where_name="Response::logExceptionToDefaultLogger",
            logdog::exception=ex);
}

void Response::with(PingResult&& resp) const {
    stream->set_code(ymod_webserver::codes::ok);
    stream->set_content_type("text", "plain");
    stream->result_body(resp.body);
}

void Response::with(EmptyResult&&) const {
    stream->set_code(ymod_webserver::codes::ok);
    stream->set_content_type("application", "json");
    stream->result_body("{}");
}

void Response::with(mail_errors::error_code&& ec) const {
    ymod_webserver::codes::code code = ymod_webserver::codes::internal_server_error;

    bool overridedErrorCode = false;

    try {
        overridedErrorCode = errorCodeToResponseCode(ec, code);
    } catch (const std::exception& ex) {
        ec = make_error(Error::unexpectedException, ex.what());
    }

    if (overridedErrorCode) {
        errorJsonResponse(code, ec);
    } else if (ec.category() == getHttpApiCategory()) {
        switch(Error(ec.value())) {
            case Error::missingParam:
            case Error::invalidArgument:
                code = ymod_webserver::codes::bad_request; break;
            case Error::wrongHttpMethod:
                code = ymod_webserver::codes::method_not_allowed; break;
            case Error::unexpectedException:
            case Error::unknownErrorCategory:
                code = ymod_webserver::codes::internal_server_error; break;
        }

        errorJsonResponse(code, ec);
    } else {
        with(make_error(Error::unknownErrorCategory, fmt::format("{}:{}", ec.category().name(), ec.full_message())));
    }
}

}
