#pragma once

#include <mail_getter/service_factory.h>
#include <mail_getter/mulcagate/http.h>
#include <mail/http_getter/client/include/module.h>

namespace mail_getter::mulcagate::http::detail {

class MulcagateHttpClientAdaptor : public mail_getter::mulcagate::http::HttpClient {
public:
    using BaseParams = mail_getter::mulcagate::http::BaseParams;
    using Response = mail_getter::mulcagate::http::Response;
    using Handler = mail_getter::mulcagate::http::ResponseHandler;

private:
    template<typename Func>
    auto getClientAndBuilder(const BaseParams& params, Func f) const {
        auto endpoint = endpoint_.format(fmt::arg("path", params.path));

        auto client = httpGetter_->create(params.requestId, nullptr);
        auto builder = ((*client).*f)(endpoint);

        return std::make_pair(std::move(client), std::move(builder));
    }
public:
    MulcagateHttpClientAdaptor(http_getter::TypedModulePtr httpGetter, http_getter::TypedEndpoint endpoint)
        : httpGetter_{std::move(httpGetter)}
        , endpoint_{std::move(endpoint)}
    { }

    void aget(const BaseParams& params, Handler handler) const override {
        auto [client, builder] = getClientAndBuilder(params, &http_getter::TypedClient::toGET);
        aop(client, std::move(builder), params, std::move(handler));
    }

    void apost(const BaseParams& params, const std::string& data, Handler handler) const override {
        auto [client, builder] = getClientAndBuilder(params, &http_getter::TypedClient::toPOST);
        aop(client, std::move(builder).body(data), params, std::move(handler));
    }

private:

    http_getter::TypedModulePtr httpGetter_;
    http_getter::TypedEndpoint endpoint_;

    template<typename Builder>
    void aop(const http_getter::TypedClientPtr& client, Builder&& builder, const BaseParams& params, Handler&& handler) const {
        using namespace http_getter::detail::operators;
        builder
            .getArgs("args"_arg=params.args)
            .headers("hdrs"_hdr=params.headers)
            .timeouts(params.timeout, params.connectTimeout)
            .keepAlive(params.keepAlive);

        client
            ->req(builder)
            ->backgroundCall("mulcagate", http_getter::Handler {
                .error=[handler](boost::system::error_code ec) {
                    handler(mail_getter::error_code(ec), Response());
                },
                .success=[handler](yhttp::response resp) {
                    handler(mail_getter::error_code(),
                            Response(static_cast<unsigned>(resp.status), std::move(resp.body)));
                    return http_getter::Result::success;
                }
            });
    }
};

inline mail_getter::mulcagate::http::HttpClientPtr getMulcagateHttpClientAdaptor(const http_getter::TypedModulePtr& httpGetter, const http_getter::TypedEndpoint& endpoint) {
    return std::make_shared<MulcagateHttpClientAdaptor>(httpGetter, endpoint);
}

}

namespace mail_getter {

mail_getter::ServiceFactoryPtr initMailStorage(const yplatform::ptree& cfg, const http_getter::TypedModulePtr& httpGetter);

}
