#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <mail/webmail/http_api_helpers/include/context.h>
#include <mail/webmail/http_api_helpers/include/ping.h>
#include <mail/webmail/http_api_helpers/include/handler_helpers.h>

#include <mail/tvm_guard/include/tvm_guard/module_impl.h>

#include <logdog/logger.h>
#include <logdog/format/tskv.h>
#include <logdog/backend/yplatform_log.h>
#include <logdog/attributes/mail_attributes.h>

#include <mail/webmail/ymod_healthcheck/include/healthcheck.h>


namespace ymod_healthcheck {

constexpr static auto formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-healthcheck-tskv-log"));

inline auto getModuleLogger(const std::string& name) {
    return ::logdog::make_log(
        formatter,
        std::make_shared<yplatform::log::source>(yplatform::log::find(name, false))
    );
}

class Response: public http_api::Response {
protected:
    void logSuccessResponse(const std::string&) const override { }
    void logErrorResponse(const std::string&) const override { }
    bool errorCodeToResponseCode(const mail_errors::error_code&, ymod_webserver::codes::code&) const override {
        return false;
    }

public:
    Response(ymod_webserver::response_ptr s)
        : http_api::Response(std::move(s))
    { }

    using http_api::Response::with;
};

Healthcheck healthcheck() {
    std::unique_lock lock(yplatform::repository::instance().mux());

    Healthcheck ret = Healthcheck::ready();

    for (const auto& [_, module]: yplatform::repository::instance().get_services()) {
        ret = Healthcheck::fold(std::move(ret), module.get_module_safe()->get_stats());
    }

    return ret;
}

http_api::PingResult alive() {
    return healthcheck().isAlive() ? http_api::PingResult::success()
                                   : http_api::PingResult::failed()
    ;
}

http_api::PingResult ready() {
    return healthcheck().isReady() ? http_api::PingResult::success()
                                   : http_api::PingResult::failed()
    ;
}

struct HttpApi: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        using namespace http_api;

        http_api::BindInfo<Response> info {
            .guarded=tvm_guard::createDummy(tvm_guard::Action::accept),
            .server=findDependency<ymod_webserver::server>(cfg, "dependencies.server"),
            .reactor=findDependency<yplatform::reactor>(cfg, "dependencies.reactor"),
        };

        http_api::bindPingHandler(info, [] (auto, auto) {
            return alive();
        });

        bindGET<http_api::PingResult>(info, "/healthcheck/alive", [] (auto, auto) {
            return alive();
        });

        bindGET<http_api::PingResult>(info, "/healthcheck/ready", [] (auto, auto) {
            return ready();
        });

        const std::string logger = cfg.get<std::string>("dependencies.logger");
        LOGDOG_(getModuleLogger(logger), notice, logdog::message="ymod_healthcheck::Module loaded");
    }
};

}

DEFINE_SERVICE_OBJECT(ymod_healthcheck::HttpApi)
