#include <mail/webmail/ymod_switchbox/include/impl/impl.h>

#include <boost/range/algorithm/transform.hpp>

namespace ymod_switchbox {

void FileAndHttp::init(const yplatform::ptree& cfg) {
    reactor_ = http_api::findDependency<yplatform::reactor>(cfg, "dependencies.reactor");

    inspectTimeout_ = yplatform::time_traits::duration_cast<std::chrono::seconds>(
        cfg.get<yplatform::time_traits::duration>("inspect_timeout")
    );

    const std::string loggerName = cfg.get<std::string>("dependencies.logger");
    logger_ = std::make_shared<ModuleLogger>(getModuleLogger(loggerName));

    http_api::BindInfo<Response> info {
        .guarded=tvm_guard::createDummy(tvm_guard::Action::accept),
        .server=http_api::findDependency<ymod_webserver::server>(cfg, "dependencies.server"),
        .reactor=reactor_,
    };

    initStrategyHandlers(cfg);
    const auto fileTimeout = cfg.get_optional<yplatform::time_traits::duration>("file_read_timeout");
    for (const auto& handler: settings_) {
        switch (handler.strategy) {
            case Strategy::http:
                bindHttpHanlders(info, handler);
                break;
            case Strategy::file:
                if (!fileTimeout) {
                    throw std::runtime_error("config must have file_read_timeout field for file strategy");
                }
                bindFileHandler(handler, yplatform::time_traits::duration_cast<std::chrono::seconds>(fileTimeout.value()));
                break;
        }
    }

    bindStatusHandler(info);
    settings_.clear();
}

void FileAndHttp::stop() {
    running_.cancel();
}

bool FileAndHttp::getValue(const std::string& name) {
    if (values_.contains(name)) {
        bool result = false;
        for (const auto& [key, val] : values_[name]) {
            result |= val;
        }
        return result;
    }
    throw std::runtime_error("there are no strategies for name: " + name);
}

void FileAndHttp::initStrategyHandlers(const yplatform::ptree& cfg) {
    boost::transform(
        cfg.equal_range("handlers"), std::inserter(settings_, settings_.end()),
        [] (auto&& val) {
            return StrategySettings {
                .strategy = yamail::data::reflection::from_string<Strategy>(val.second.template get<std::string>("strategy")),
                .name = val.second.template get<std::string>("name"),
                .path = val.second.template get<std::string>("path"),
            };
        }
    );

    for (auto it = settings_.begin(); it != settings_.end(); ++it) {
        if (values_.contains(it->name) && values_[it->name].contains(it->strategy)) {
            throw std::runtime_error("config cannot have two handlers for one name with the same strategy");
        }
        values_[it->name].emplace(it->strategy, false);
    }
}

void FileAndHttp::bindHttpHanlders(const http_api::BindInfo<Response>& info, const StrategySettings& settings) {
    bindPOST<http_api::EmptyResult>(info, "/ymod_switchbox/" + settings.path + "/enable", [=] (auto, auto) {
        changeValueForStrategy(true, settings.name, settings.strategy);
        return http_api::EmptyResult();
    });

    bindPOST<http_api::EmptyResult>(info, "/ymod_switchbox/" + settings.path + "/disable", [=] (auto, auto) {
        changeValueForStrategy(false, settings.name, settings.strategy);
        return http_api::EmptyResult();
    });

    LOGDOG_(*logger_, notice, logdog::message="ymod_switchbox::HttpChanger for " + settings.name + " loaded");
}

void FileAndHttp::bindFileHandler(const StrategySettings& settings, const yplatform::time_traits::duration& fileTimeout) {
    const auto timer = std::make_shared<Timer>(*reactor_->io());
    FileObserver observer(shared_from(this), std::move(timer), settings, fileTimeout);
    observer();
        
    LOGDOG_(*logger_, notice, logdog::message="ymod_switchbox::FileChanger for " + settings.name + " loaded");
}

void FileAndHttp::bindStatusHandler(const http_api::BindInfo<Response>& info) {
    bindGET<StatusResult>(info, "/ymod_switchbox/status", [this] (auto, auto) {
        StatusResult result;
        for (const auto& [name, value] : this->values_) {
            result.values[name] = getValue(name);
        }
        return result;
    });
}

void FileAndHttp::asyncWaitExpectedValue(const std::string& name, bool expectedValue, OnValue hook) {
    auto timer = std::make_shared<Timer>(*reactor_->io());
    asyncWaitExpectedValue(name, expectedValue, std::move(timer), std::move(hook));
}

bool FileAndHttp::changeValueForStrategy(bool newValue, const std::string& name, Strategy strategy) {
    if (values_.contains(name) && values_[name].contains(strategy)) {
        values_[name][strategy] = newValue;
        return true;
    }
    return false;
}

void FileAndHttp::FileObserver::operator() (const mail_errors::error_code& /*e*/) {
    try {
        reenter(this) {
            while (!ctx->self->running_.cancelled()) {
                if (std::filesystem::exists(ctx->settings.path)) {
                    ctx->self->changeValueForStrategy(true, ctx->settings.name, ctx->settings.strategy);
                } else {
                    ctx->self->changeValueForStrategy(false, ctx->settings.name, ctx->settings.strategy);
                }
                ctx->timer->expires_after(ctx->timeout);
                yield ctx->timer->async_wait(*this);
            }
        }
    } catch (const std::exception& e) {
        LOGDOG_(ctx->self->logger_, error, logdog::exception=e);
    } catch (...) {
        LOGDOG_(ctx->self->logger_, error, logdog::message="fileHandler: unexpected unknown exception");
    }
}

void FileAndHttp::asyncWaitExpectedValue(const std::string& name, bool expectedValue, TimerPtr timer, OnValue hook) {
    const auto currentValue = getValue(name);
    if (currentValue != expectedValue && !running_.cancelled()) {
        timer->expires_after(inspectTimeout_);
        timer->async_wait([=] (const boost::system::error_code& ec) {
            if (ec != boost::asio::error::operation_aborted) {
                asyncWaitExpectedValue(name, expectedValue, timer, hook);
            }
        });
    } else {
        hook(mail_errors::error_code(), expectedValue);
    }
}

}
