#pragma once

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <mail/webmail/http_api_helpers/include/context.h>
#include <mail/webmail/http_api_helpers/include/handler_helpers.h>

#include <mail_errors/error_code.h>
#include <mail/tvm_guard/include/tvm_guard/module_impl.h>

#include <mail/webmail/ymod_switchbox/include/module.h>
#include <mail/webmail/ymod_switchbox/include/response.h>
#include <mail/webmail/ymod_switchbox/include/types.h>
#include <mail/webmail/ymod_switchbox/include/config.h>
#include <mail/webmail/ymod_switchbox/include/task_context_holder.h>

#include <logdog/logger.h>
#include <logdog/format/tskv.h>
#include <logdog/backend/yplatform_log.h>

#include <boost/asio/yield.hpp>

namespace ymod_switchbox {

constexpr static auto formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-switchbox-tskv-log"));

inline auto getModuleLogger(const std::string& name) {
    return ::logdog::make_log(
        formatter,
        std::make_shared<yplatform::log::source>(yplatform::log::find(name, false))
    );
}

using ModuleLogger = decltype(getModuleLogger(""));

struct FileAndHttp : public ModuleProvider, public yplatform::module {
    using Timer = boost::asio::steady_timer;
    using TimerPtr = std::shared_ptr<Timer>;
    using SelfPtr = std::shared_ptr<FileAndHttp>;

    virtual ~FileAndHttp() = default;

    void init(const yplatform::ptree& cfg);
    void stop();
    bool getValue(const std::string& name) override;
    
protected:
    void initStrategyHandlers(const yplatform::ptree& cfg);
    void bindHttpHanlders(const http_api::BindInfo<Response>& info, const StrategySettings& settings);
    void bindFileHandler(const StrategySettings& settings, const yplatform::time_traits::duration& fileTimeout);
    void bindStatusHandler(const http_api::BindInfo<Response>& info);
    void asyncWaitExpectedValue(const std::string& name, bool expectedValue, OnValue hook) override;
    bool changeValueForStrategy(bool newValue, const std::string& name, Strategy strategy);

    struct FileObserver : boost::asio::coroutine {
        struct Ctx {
            SelfPtr self;
            TimerPtr timer;
            StrategySettings settings;
            yplatform::time_traits::duration timeout;
        };

        std::shared_ptr<Ctx> ctx;

        FileObserver(SelfPtr self, TimerPtr timer, StrategySettings settings, yplatform::time_traits::duration timeout)
        : ctx(std::make_shared<Ctx>(Ctx{std::move(self), std::move(timer), std::move(settings), timeout})) {}

        void operator() (const mail_errors::error_code& /*e*/ = mail_errors::error_code());
    };

private:
    void asyncWaitExpectedValue(const std::string& name, bool expectedValue, TimerPtr timer, OnValue hook);

    std::shared_ptr<yplatform::reactor> reactor_;
    std::shared_ptr<ModuleLogger> logger_;
    yplatform::time_traits::duration inspectTimeout_;
    std::vector<StrategySettings> settings_;
    TaskContextHolder running_;
    std::map<std::string, std::map<Strategy, std::atomic_bool>> values_;
};

}
