# -*- coding: utf-8 -*-
from __future__ import print_function

import sys
from argparse import ArgumentParser
from collections import defaultdict
from contextlib import closing

import jinja2
import yaml

import codecs

try:
    import jsonpath_rw as JSPATH
except ImportError:
    print('''Install jsonpath-rw lib
sudo apt-get install -y python-jsonpath-rw
''', file=sys.stderr)
    raise


SILENT = False


def colorify(text, color):
    end = '\033[0m'
    colors = dict()
    colors["blue"] = '\033[94m'
    colors["green"] = '\033[92m'
    colors["yellow"] = '\033[33m'
    colors["red"] = '\033[91m'
    colors["cyan"] = '\033[36m'
    colors["violent"] = '\033[35;1m'

    if color not in colors:
        raise Exception("No such color: "+color)
    return colors[color]+text+end


def log(s):
    if not SILENT:
        print(colorify(str(s)+'\n', 'cyan'))


@jinja2.contextfilter
def find_first_filter(context, value):
    js_path = JSPATH.parse(value)
    found = [f.value for f in js_path.find(context)]
    if not found:
        return context.environment.undefined(
            'No values by path %r found' % value)
    return found[0]


FILTERS = dict(
    find_first=find_first_filter,
)


def make_environment():
    environment = jinja2.Environment(
        undefined=jinja2.StrictUndefined
    )
    environment.filters.update(FILTERS)
    return environment


def render(text, salt):
    log(salt)
    template = make_environment().from_string(text.decode('utf-8'))
    return template.render(**salt)


def get_dst(src, target, target_before_extension=False, sre_format=False):
    if target_before_extension:
        return get_dst_with_target_before_extension(src, target)
    if sre_format:
        directory, extension = src.split("*.")
        return directory + target + '.' + extension
    return src + '-' + target


def get_dst_with_target_before_extension(src, target):
    prefix, suffix = src.rsplit('.', 1)
    return '%s-%s.%s' % (prefix, target, suffix)


def read_config_file(name):
    retval = {}
    with closing(open(name)) as fd:
        retval = yaml.full_load(fd)
        assert not retval or isinstance(retval, dict), \
            'Got unexpected common type: %r , %r ' % (
                type(retval), retval)
    return retval


def make_config(target, template, *configs, **kwargs):
    global SILENT
    if 'silent' in kwargs:
        SILENT = kwargs['silent']

    config = defaultdict(dict)
    for c in configs:
        config.update(c[target])

    return render(template, config)


def skip_environment(args, target):
    return target in args.skip or (len(args.generate_for) > 0 and target not in args.generate_for)


def make_config_from_command_line():
    global SILENT  # pylint: disable=W0603
    parser = ArgumentParser()
    parser.add_argument(
        'src',
        help='source template'
    )
    parser.add_argument(
        '--cfg',
        help='files with common config keys',
        nargs='*',
        default=['common.yaml']
    )
    parser.add_argument(
        '--out',
        help='output file prefix'
    )
    parser.add_argument(
        '--silent', action='store_true',
        help='make output silent again'
    )
    environments = parser.add_mutually_exclusive_group()
    environments.add_argument(
        '--skip',
        default=[],
        nargs='*',
        help='don\'t generate configs for this environment'
    )
    environments.add_argument(
        '--generate_for',
        default=[],
        nargs='*',
        help='generate only for these environments'
    )
    output_format = parser.add_mutually_exclusive_group()
    output_format.add_argument(
        '--target_before_extension',
        default=False,
        action='store_true',
        help='insert target before file extension like "file-target.ext"',
    )
    output_format.add_argument(
        '--sre_format',
        default=False,
        action='store_true',
        help='Format for SRE naming (MAILSRE-699). e.g.: /path/to/generate/*.yaml',
    )

    args = parser.parse_args()
    SILENT = args.silent

    configs = [read_config_file(cfg) for cfg in args.cfg]

    common = defaultdict(dict)
    for cfg in configs:
        for target in cfg:
            if not skip_environment(args, target):
                common[target].update(cfg[target])

    with closing(codecs.open(args.src, encoding='utf-8')) as src_fd:
        src = src_fd.read()

    log(src)

    for target in common:
        if skip_environment(args, target):
            print(colorify('skip %s key' % target, 'yellow'))
            continue
        config = common[target]

        dst = get_dst(args.out or args.src, target, target_before_extension=args.target_before_extension, sre_format=args.sre_format)
        print(colorify('Generating for target: '+target, 'yellow'))

        with closing(codecs.open(dst, mode='w', encoding='utf-8')) as dst_fd:
            try:
                dst_fd.write(render(src, config))
            except jinja2.exceptions.UndefinedError as exc:
                print(
                    colorify(
                        "Can't find required key in %s environment: %s" % (target, exc),
                        'red'
                    ),
                    file=sys.stderr)
                raise
