#!/bin/bash

set -ex


function generate_version() {
    local TAG="$1"
    local PROJECTS="$2"
    local PREFIX="$3"

    local ENV_PREFIX='mail.wmi-qa'
    [[ -n "$4" ]] && ENV_PREFIX="$4"
    local ENV_POSTFIX=`echo $PROJECTS | sed -r -e 's|\s+|-|g' -e 's|akita|a|' -e 's|barbet|b|' -e 's|spaniel|sp|' -e 's|hound|h|' -e 's|mops|m|' -e 's|sendbernar|s|' -e 's|mbody|mb|' -e 's|retriever|r|'`
    ENV_VERSION=`echo $TAG | tr -s '.' '-' | tr -s '_' '-' | tr -s '~' '-' | tr '[:upper:]' '[:lower:]'`

    ENV_VERSION="y-$PREFIX$ENV_VERSION-$ENV_POSTFIX"
    if (( $(wc -m <<< "$ENV_VERSION") > 63 )); then
        ENV_VERSION="${ENV_VERSION:0:63}";
    fi

    ENV_NAME="$ENV_PREFIX.$ENV_VERSION"
}

function add_proj() {
    function image_hash {
        curl -s "https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl=$1&tag=$2" | $JQ -r '.hash'
    }

    local IMAGE_TAG="$2"
    local REGISTRY='registry.yandex.net/mail'
    local SENDBERNAR_IMG="$REGISTRY/sendbernar/sendbernar"
    local HOUND_IMG="$REGISTRY/hound/hound"
    local AKITA_IMG="$REGISTRY/akita/akita"
    local BARBET_IMG="$REGISTRY/barbet/barbet"
    local SPANIEL_IMG="$REGISTRY/spaniel/spaniel"
    local QUEUEDB_IMG="$REGISTRY/queuedb/queuedb"
    local SPANIELDB_IMG="$REGISTRY/spanieldb/spanieldb"
    local MOPS_IMG="$REGISTRY/mops/mops"
    local MOPSDB_IMG="$REGISTRY/mopsdb/mopsdb"
    local RETRIEVER_IMG="$REGISTRY/retriever/retriever"
    local MBODY_IMG="$REGISTRY/mbody/mbody"

    case $1 in
        akita)
            local self_img="$AKITA_IMG"
            ;;
        barbet)
            add_proj queuedb $IMAGE_TAG
            local self_img="$BARBET_IMG"
            ;;
        spaniel)
            add_proj queuedb $IMAGE_TAG
            add_proj spanieldb $IMAGE_TAG
            local self_img="$SPANIEL_IMG"
            ;;
        queuedb)
            local self_img="$QUEUEDB_IMG"
            ;;
        spanieldb)
            local self_img="$SPANIELDB_IMG"
            ;;
        hound)
            local self_img="$HOUND_IMG"
            ;;
        sendbernar)
            local self_img="$SENDBERNAR_IMG"
            ;;
        mops)
            add_proj mopsdb $IMAGE_TAG
            local self_img="$MOPS_IMG"
            ;;
        mopsdb)
            local self_img="$MOPSDB_IMG"
            ;;
        retriever)
            local self_img="$RETRIEVER_IMG"
            ;;
        mbody)
            local self_img="$MBODY_IMG"
            ;;
    esac

    local self_hash="$(image_hash $self_img $IMAGE_TAG)"

    WMIQA_CMD="$WMIQA_CMD \"$1 $1 $self_img:$IMAGE_TAG $self_hash\""
}

function copy_project_from_production() {
    local SERVICE="$1"
    local TOKEN="$2"

    if [ "$SERVICE" == "mops" ]; then
        add_proj_for_smtp_gate "mopsdb" "$TOKEN"
    fi

    local HOST="https://platform.yandex-team.ru"
    local AUTH="authorization: OAuth $TOKEN"


    local RESP=`curl -v --fail -H "$AUTH" "$HOST/api/v1/environment/dump/mail.$SERVICE.production" | $JQ '.components[] | select(.componentName=="'$SERVICE'") | .properties | "\(.repository) \(.hash)"' -r`
    WMIQA_CMD="$WMIQA_CMD \"$SERVICE $SERVICE $RESP\""
}

function create_env_and_wait() {
    function get_host() {
        (curl   -s -H "$AUTH" "$HOST/api/v1/runtime/$ENV_NAME.$1/$Q_VERSION/" | $JQ '.runningInstances[0].host' -r 2>/dev/null) || echo "$2"
    }

    local ENV_NAME="$1"
    local ENV_VERSION="$2"
    local TOKEN="$3"
    local OUTPUT="$4"

    local HOST="https://platform.yandex-team.ru"
    local AUTH="authorization: OAuth $TOKEN"

    curl -v --fail -H "$AUTH" "$HOST/api/v1/environment/upload" --data-binary "@data.json" -H "content-type: application/json"


    while true; do
        local status="$(curl -s -H "$AUTH" "$HOST/api/v1/environment/stable/$ENV_NAME" | $JQ -r '.status')"

        case "$status" in
            DEPLOYED )
                break;
                ;;
            * )
                echo "Environment's status is \'$status\'"
                ;;
        esac
        sleep 10
    done;


    local Q_VERSION=`curl -s -H "$AUTH" "$HOST/api/v1/environment/stable/$ENV_NAME" | $JQ '.version'`

    local HOST_SBR="$(get_host sendbernar sendbernar-test.mail.yandex.net)"
    local HOST_HOUND="$(get_host hound meta-test.mail.yandex.net)"
    local HOST_AKITA="$(get_host akita akita-test.mail.yandex.net)"
    local HOST_BARBET="$(get_host barbet barbet-test.mail.yandex.net)"
    local HOST_SPANIEL="$(get_host spaniel spaniel-test.mail.yandex.net)"
    local HOST_MOPS="$(get_host mops mops-test.mail.yandex.net)"
    local HOST_RETRIEVER="$(get_host retriever webattach-test.mail.yandex.net)"
    local HOST_MBODY="$(get_host mbody mbody-test.mail.yandex.net)"

    echo "HOST_SENDBERNAR=$HOST_SBR"        >  $OUTPUT
    echo "HOST_HOUND=$HOST_HOUND"           >> $OUTPUT
    echo "HOST_AKITA=$HOST_AKITA"           >> $OUTPUT
    echo "HOST_BARBET=$HOST_BARBET"         >> $OUTPUT
    echo "HOST_SPANIEL=$HOST_SPANIEL"       >> $OUTPUT
    echo "HOST_MOPS=$HOST_MOPS"             >> $OUTPUT
    echo "HOST_MBODY=$HOST_MBODY"           >> $OUTPUT
    echo "HOST_RETRIEVER=$HOST_RETRIEVER"   >> $OUTPUT
    echo "ZK_ROOT_NODE=" >> $OUTPUT
}
