import {Address, Attachment, Body, FileType, Folder, Label, Mailbox, Message, MimeType} from "./model";
import {colorImage, NameSize, randomColor, randomEmail, randomPlainContent, randomSubject, uniqueName} from "./utils";

export class LargeMailbox extends Mailbox {

    private readonly LABELS_COUNT = 993;
    private readonly FOLDERS_COUNT = 992;
    private readonly MESSAGES_PER_FOLDER_COUNT = 100;
    private readonly ATTACHMENT_PER_MESSAGE_COUNT = 10;

    private readonly labelsCache: Label[] = [];

    private readonly autotestFolders: string[] = ["olyd", "test_folder", "SwitchAccFolder", "lalafolder",
        "папка0123folder", "المتحد!@#$%&*^中华人民共和国"];
    private readonly autotestFolderWithSubfolders = "meduza";
    private readonly autotestSubFolders: string[] = ["somalia", "chiberia"];
    private readonly autotestLabels: string[] = ["autotest", "espresso", "cafe-con-leche", "epic", "olala",  "kilo",
        "dota2", "wow"];

    constructor(seed: string = null) {
        super(true, seed);

        for(let n=0; n<this.autotestLabels.length; n++) {
            let name = this.autotestLabels[n];
            this.labelsCache.push(this.createUserLabel(name, randomColor(super.random())));
        }
        for(let n=0; n<this.LABELS_COUNT; n++) {
            this.labelsCache.push(this.createUserLabel(uniqueName(), randomColor(super.random())));
        }
    }

    protected folderMessages(folder: Folder): Message[] {
        let result = [];
        for(let n=0; n<this.MESSAGES_PER_FOLDER_COUNT; n++) {
            let attach = [];
            for(let k=0; k<this.ATTACHMENT_PER_MESSAGE_COUNT; k++) {
                attach.push(new Attachment({
                    hid: `1.${k + 3}`,
                    diplayName: uniqueName(NameSize.SHORT),
                    fileType: FileType.IMAGE,
                    mime: MimeType.PNG,
                    url: colorImage(100, 100, randomColor(super.random()).hex(), `${k}.png`)
                }));
            }
            result.push(new Message({
                folder: folder,
                subject: randomSubject(super.random()),
                from: new Address(randomEmail(super.random())),
                to: [new Address(randomEmail(super.random()))],
                labels: [this.labelsCache[n % this.LABELS_COUNT]],
                bodies: [new Body("1.2", randomPlainContent(super.random()), MimeType.PLAIN)],
                attachments: attach
            }))
        }
        return result;
    }

    protected labels(): Label[] {
        return this.labelsCache;
    }

    protected userFolders(): Folder[] {
        let result = [];
        let previous = this.root;

        let autotestFolderWithSubfolder = this.createUserFolderInRoot(this.autotestFolderWithSubfolders);
        result.push(autotestFolderWithSubfolder);

        for (const name of this.autotestFolders) {
            result.push(this.createUserFolderInRoot(name));
        }

        for(const name of this.autotestSubFolders) {
            result.push(this.createUserFolder(name, autotestFolderWithSubfolder));
        }

        for(let n=0; n<this.FOLDERS_COUNT; n++) {
            let name = uniqueName(NameSize.SHORT);
            if (n % 2 == 0) {
                let current = this.createUserFolder(name, previous);
                result.push(current);
                previous = current;
            } else {
                result.push(this.createUserFolder(name, this.root));
            }
        }
        return result;
    }

}
