import * as random from "random";
import * as seedrandom from "seedrandom";
import {MobileAPI, urlToMobileAPIHandler} from "../poisons/mail/handlers";
import {generateID, timestamp, utcTimestamp} from "./utils"
import {Request} from "../poisons/patch";
import {URLSearchParams} from "url";

const OK = 1;
const ERROR = 999;

export enum FolderType {
    INBOX = 1,
    USER,
    OUTBOX,
    SENT,
    DRAFT,
    SPAM,
    TRASH,
    ARCHIVE,
    TEMPLATES,
    DISCOUNTS,
    UNKNOWN,
    UNSUBSCRIBE
}

export enum TreadingType {
    YES = 1,
    NO = 0,
    GLOBAL = "?"
}

export enum NotifyType {
    YES = 1,
    NO = 0,
    GLOBAL = "?"
}

export class Folder {
    readonly id: string;
    readonly parent: Folder | null;
    readonly displayName: string;
    readonly unread: number;
    readonly count: number;
    readonly type: FolderType;
    readonly treaded: TreadingType;
    readonly notify: NotifyType;
    readonly position: number;

    constructor(params: {
        id?: string;
        parent?: Folder | null;
        displayName?: string;
        unread?: number;
        count?: number;
        type?: FolderType;
        treaded?: TreadingType;
        notify?: NotifyType;
        position?: number;

    }) {
        this.id = params.id ||  generateID();
        this.parent = params.parent;
        this.displayName = params.displayName || this.id;
        this.type = params.type || FolderType.USER;
        this.unread = params.unread || 0;
        this.count = params.count || 0;
        this.treaded = params.treaded || TreadingType.NO;
        this.notify = params.notify || NotifyType.NO;
        this.position = params.position || 0;
    }

}

export enum LabelType {
    USER = 1,
    SOCIAL,
    SYSTEM,
    FAKE_SEEN,
    FAKE_ATTACHED,
    IMPORTANT = 6
}

export class Color {
    static readonly RED: Color = new Color(255, 0, 0);
    static readonly GREEN: Color = new Color(0, 255, 0);
    static readonly BLUE: Color = new Color(0, 0, 255);

    private readonly red: number;
    private readonly green: number;
    private readonly blue: number;

    constructor(red: number, green: number, blue: number) {
        this.red = red;
        this.green = green;
        this.blue = blue;
    }

    private static i2hs(n: number): string {
        const result = n.toString(16).toUpperCase();
        return result.length == 1? "0" + result: result;
    }

    hex(): string {
        return `${Color.i2hs(this.red)}${Color.i2hs(this.green)}${Color.i2hs(this.blue)}`;
    }


}

export class Label {
    readonly id: string;
    readonly displayName: string;
    readonly color: Color;
    readonly count: number;
    readonly unread: number;
    readonly type: LabelType;

    constructor(params: {
        id?: string;
        displayName?: string;
        color?: Color;
        count?: number;
        unread?: number;
        type?: LabelType;

    }) {
        const colors = [Color.RED, Color.GREEN, Color.BLUE];
        this.id = params.id || generateID();
        this.displayName = params.displayName || this.id;
        this.color = params.color || colors[Math.floor(Math.random() * colors.length)];
        this.count = params.count || 0;
        this.unread = params.unread || 0;
        this.type = params.type || LabelType.USER;
    }
}

export class Thread {
    readonly id: string;
}

export enum MessageStatus {
    READ = 0,
    UNREAD,
    ANSWERED,
    FORWARDED
}

export enum SubjectPrefix {
    REPLY = "Re:",
    FOWWARD = "Fwd:",
    NONE = ""
}

export class Address {
    readonly email: string;
    readonly name: string;

    constructor(email: string, name: string = null) {
        this.email = email;
        this.name = name || email;
    }
}

enum AddressType {
    TO = 1,
    FROM,
    CC,
    BCC,
    REPLY_TO
}

export enum MessageType {
    DELIVERY_REPORT = 1,
    REGISTRATION_NOTIFICATION,
    SOCIAL_NETWORK,
    PEOPLE,
    TICKETS,
    ONLINE_STORES,
    NOTIFICATIONS,
    //  8 - bounces, 9 - official, 10 - scripts, 11 - dating services, 12 - greetings, 13 - news,
    //  14 - groupons, 15 - dating services, 16 - airplane tickets, 17 - emails from banks,
    //  18 - social networks, 19 - travel agencies, 20 - railway tickets, 21 - realty,
    //  22 - personal news, 23 - online stores, 24 - emails from companies, 25 - job sites,
    //  26 - games sites, 27 - schema, 28 - cancel, 29 - technical, 30 - media, 31 - ads,
    //  32 - emails from internet providers, 33 - forums, 34 - mobile, 35 - hotel reservations
}

export enum MimeType {
    PLAIN = "text/plain",
    HTML = "text/html",
    GIF = "image/gif",
    JPEG = "image/jpeg",
    PNG = "image/png"
}

export enum FileType {
    UNKNOWN = "unknown",
    GENERAL = "general",
    ZIP = "zip",
    RAR = "rar",
    XLS = "xls",
    PDF = "pdf",
    PPT = "ppt",
    QT = "qt",
    DOC = "doc",
    IMAGE = "image",
    MAIL = "mail",
    VIDEO = "video",
    AUDIO = "audio",
    APPLICATION = "application"
}

export class Attachment {
    readonly hid: string;
    readonly displayName: string;
    readonly fileType: FileType;
    readonly fromDisk: boolean;
    readonly size: number;
    readonly inline: boolean;
    readonly mime: MimeType;
    readonly preview: boolean;
    readonly url: string;

    constructor(params: {
        hid: string;
        diplayName: string;
        fileType: FileType;
        fromDisk?: boolean;
        size?: number;
        inline?: boolean;
        mime: MimeType;
        preview?: boolean;
        url: string;
    }) {
        this.hid = params.hid;
        this.displayName = params.diplayName;
        this.fileType = params.fileType;
        this.fromDisk = params.fromDisk || false;
        this.size = params.size || 0;
        this.inline = params.inline || false;
        this.mime = params.mime;
        this.preview = params.preview || false;
        this.url = params.url;
    }
}

export type BodyType = MimeType.PLAIN | MimeType.HTML;

export class Body {
    readonly hid: string;
    readonly content: string;
    readonly contentType: BodyType;

    constructor(hid: string, content: string, type: BodyType) {
        this.hid = hid;
        this.content = content;
        this.contentType = type;
    }
}

export class Message {
    readonly id: string;
    readonly rfcID: string | null;
    readonly folder: Folder;
    // readonly thread: Thread;
    readonly labels: ReadonlySet<Label>;
    readonly status: ReadonlySet<MessageStatus>;
    // readonly threadCount: number;
    // readonly scn: number;
    readonly timestamp: number;
    readonly utcTimestamp: number;
    readonly subjectPrefix: SubjectPrefix;
    readonly subject: string;
    readonly from: Address;
    readonly to: ReadonlySet<Address>;
    readonly cc: ReadonlySet<Address>;
    readonly bcc: ReadonlySet<Address>;
    readonly replyTo: Address | null;
    readonly firstLine: string;
    readonly messageTypes: ReadonlySet<MessageType>;
    readonly type: MimeType.PLAIN | MimeType.HTML;
    readonly attachments: ReadonlyMap<string, Attachment>;
    readonly body: ReadonlyMap<BodyType, Body>;

    constructor(params: {
        id?: string;
        rfcID?: string;
        folder: Folder;
        labels?: Label[];
        status?: MessageStatus[];
        timestamp?: number;
        utcTimestamp?: number;
        subjectPrefix?: SubjectPrefix;
        subject: string;
        from: Address;
        to: Address[];
        cc?: Address[];
        bcc?: Address[];
        replyTo?: Address | null;
        firstLine?: string;
        messageTypes?: MessageType[];
        attachments?: Attachment[];
        bodies: Body[];
    }) {
        this.id = params.id || generateID();
        this.rfcID = params.id || `<${this.id}@wmi-dev2.mail.yandex.net>`;
        this.folder = params.folder;
        this.labels = new Set<Label>(params.labels) || new Set<Label>();
        this.status = new Set<MessageStatus>(params.status) || new Set<MessageStatus>([MessageStatus.UNREAD]);
        this.timestamp = params.timestamp || timestamp();
        this.utcTimestamp = params.timestamp || utcTimestamp();
        this.subjectPrefix = params.subjectPrefix || SubjectPrefix.NONE;
        this.subject = params.subject;
        this.from = params.from;
        this.to = new Set<Address>(params.to) || new Set<Address>();
        this.cc = new Set<Address>(params.cc) || new Set<Address>();
        this.bcc = new Set<Address>(params.bcc) || new Set<Address>();
        this.replyTo = params.replyTo || null;
        this.firstLine = params.firstLine || "";
        this.messageTypes = new Set<MessageType>(params.messageTypes) || new Set<MessageType>([MessageType.PEOPLE]);
        let attachments = new Map<string, Attachment>();
        for(let attachment of params.attachments || new Array<Attachment>()) {
            attachments.set(attachment.hid, attachment);
        }
        this.attachments = attachments;
        let bodies: Map<BodyType, Body> = new Map<BodyType, Body>();
        for(let body of params.bodies) {
            bodies.set(body.contentType, body);
        }
        this.body = bodies;
    }

}

export abstract class Mailbox {
    readonly inbox: Folder;
    readonly outbox: Folder;
    readonly sent: Folder;
    readonly draft: Folder;
    readonly spam: Folder;
    readonly trash: Folder;
    readonly archive: Folder;
    readonly root: Folder | null;
    readonly important: Label;

    private readonly rng;
    private readonly roots: Folder[] = [];
    private position = 0;
    private revision = 0; // TODO: should calculate this
    private md5 = "f5e0606f5d7382a9b606da39986437f1"; // TODO: should calculate this
    private staticResponse = null;
    private readonly fidToMessages: Map<String, Message[]> = new Map();
    private readonly midToMessage: Map<String, Message> = new Map();
    private readonly lidToLabel: Map<String, Label> = new Map();

    protected constructor(create_root: boolean = true, seed: string) {
        this.inbox = new Folder({displayName: "Inbox", type: FolderType.INBOX, position: this.position++});
        this.outbox = new Folder({displayName: "Outbox", type: FolderType.OUTBOX, position: this.position++});
        this.sent = new Folder({displayName: "Sent", type: FolderType.SENT, position: this.position++});
        this.draft = new Folder({displayName: "Drafts", type: FolderType.DRAFT, position: this.position++});
        this.spam = new Folder({displayName: "Spam", type: FolderType.SPAM, position: this.position++});
        this.trash = new Folder({displayName: "Trash", type: FolderType.TRASH, position: this.position++});
        this.archive = new Folder({displayName: "Archive", type: FolderType.ARCHIVE, position: this.position++});
        this.roots.push(this.inbox, this.outbox, this.sent, this.draft, this.spam, this.trash, this.archive);
        this.important = new Label({displayName: "priority_high", type: LabelType.IMPORTANT});
        if (create_root) {
            this.root = new Folder({displayName: "Папки", type: FolderType.USER, position: this.position++});
            this.roots.push(this.root);
        } else {
            this.root = null;
        }
        if (seed) {
            // TODO: something strange is going on here: without first line second doesn't work :(
            random.use(seedrandom(seed));
            this.rng = random.clone(seedrandom(seed));
            random.use(Math.random);
        }
    }

    protected random() {
        return this.rng;
    }

    protected createUserFolder(displayName: string, parent: Folder = this.root) {
        displayName = parent.displayName + "|" + displayName;
        return new Folder({displayName: displayName, parent: parent, type: FolderType.USER});
    }

    protected createUserFolderInRoot(displayName: string) {
        return new Folder({displayName:displayName, type: FolderType.USER, position: this.position++});
    }

    protected createUserLabel(displayName: string, color?: Color) {
        return new Label({displayName: displayName, color: color, type: LabelType.USER});
    }

    protected abstract userFolders(): Folder[];

    protected abstract folderMessages(folder: Folder): Message[];

    protected labels(): Label[] {
        return [];
    }

    private static serializeFolder(folder: Folder) {
        return {
            "fid": folder.id,
            "parent": folder.parent? folder.parent.id: "",
            "display_name": folder.displayName,
            "count_unread": folder.unread,
            "count_all": folder.count,
            "type": folder.type,
            "options":{
                "threaded": folder.treaded,
                "notify": folder.notify,
                "position": String(folder.position)
            }
        }
    }

    private static serializeLabel(label: Label) {
        return {
            "lid": label.id,
            "display_name": label.displayName,
            "color": label.color.hex(),
            "count_unread": String(label.unread),
            "count_all": String(label.count),
            "type": label.type
        }
    }

    private static serializeRecipients(message: Message, serializeFrom: boolean = false) {
        const serializer = function(address: Address, type: AddressType) {
            return {
                "email": address.email,
                "name": address.name,
                "type": type
            }
        };
        return [].concat(
            serializeFrom? [serializer(message.from, AddressType.FROM)]: [],
            Array.from(message.to).map(address => serializer(address, AddressType.TO)),
            Array.from(message.cc).map(address => serializer(address, AddressType.CC)),
            Array.from(message.bcc).map(address => serializer(address, AddressType.BCC)),
            message.replyTo? [serializer(message.replyTo, AddressType.REPLY_TO)]: []
        )
    }

    private static serializeMessage(message: Message, threaded = true) {
        let result = {
            "mid": message.id,
            "fid": message.folder.id,
            "tid": message.id,
            "lid": Array.from(message.labels).map(label => label.id),
            "modseq": "",
            "status": Array.from(message.status),
            "scn": 1,
            "timestamp": String(message.timestamp),
            "utc_timestamp": String(message.timestamp),
            "hasAttach": message.attachments.size > 0,
            "subjEmpty": message.subject.length > 0,
            "subjPrefix": message.subjectPrefix,
            "subjText": message.subject,
            "from": {
                "email": message.from.email,
                "name": message.from.name,
                "type": AddressType.FROM
            },
            "recipients": Mailbox.serializeRecipients(message),
            "firstLine": message.firstLine,
            "types": Array.from(message.messageTypes),
            "tab": "default"
        };
        if (threaded) {
            result["threadCount"] = 1
        }
        return result;
    }

    private static serializeAttachment(attachment: Attachment) {
        return {
            "hid": attachment.hid,
            "display_name": attachment.displayName,
            "class": attachment.fileType,
            "narod": attachment.fromDisk,
            "size": attachment.size,
            "is_inline": attachment.inline,
            "mime_type": attachment.mime,
            "preview_supported": attachment.preview,
            "download_url": attachment.url
        }
    }

    private static serializeBody(body: Body) {
        return {
            "hid": body.hid,
            "content": body.content,
            "content_type": body.contentType,
            "facts":"{\"events\":[]}"
        }
    }

    private static serializeMessageBody(message: Message) {
        return {
            "status": {
                "status": OK,
            },
            "info": {
                "mid": message.id,
                "ext_msg_id": message.rfcID,
                "references":"",
                "recipients": Mailbox.serializeRecipients(message, true),
                "attachments": Array.from(message.attachments.values()).map(Mailbox.serializeAttachment),
                "dkim": {
                    "status": "OK",
                    "domain": "yandex.ru"
                }
            },
            "body": Array.from(message.body.values()).map(Mailbox.serializeBody)
        }
    }

    private initStaticResponses(userFolders, labels) {
        if (!this.staticResponse) {
            this.staticResponse = {};
            let xlist: any = [
                {
                    "status": {
                        "status": OK
                    },
                    "md5": this.md5,
                    "mailbox_revision": String(this.revision)
                }
            ];
            xlist.push(...this.roots.map(Mailbox.serializeFolder));
            xlist.push(...userFolders.map(Mailbox.serializeFolder));
            xlist.push(...labels.map(Mailbox.serializeLabel));
            this.staticResponse[MobileAPI.XLIST] = xlist;
        }
    }

    private init() {
        const labels = this.labels().concat(this.important);
        for(let label of labels) {
            this.lidToLabel.set(label.id, label);
        }
        const userFolders = this.userFolders();
        this.initStaticResponses(userFolders, labels);
        if (this.fidToMessages.size == 0 && this.midToMessage.size == 0) {
            for (let folder of [].concat(this.roots, userFolders)) {
                const folderMessages = this.folderMessages(folder);
                this.fidToMessages.set(folder.id, folderMessages);
                for(let message of folderMessages) {
                    this.midToMessage.set(message.id, message);
                }
            }
        }
    }

    private messages(request: Request): any[] {
        let result: any = [];
        const fid = "fid";
        const tid = "tid";
        const lid = "lid";
        for(let fr of request.body["requests"]) {
            let ms = [];
            if (fid in fr) {
                let id = fr[fid];
                if (this.fidToMessages.has(id)) {
                    ms = this.fidToMessages.get(id).slice(fr["first"], fr["last"]);
                } else {
                    console.log(`Cannot find fid ${id} for messages`);
                }
            }
            if (tid in fr) {
                let id = fr[tid];
                if (this.midToMessage.has(id)) {
                    ms = [this.midToMessage.get(id)];
                } else {
                    console.log(`Cannot find tid ${id} for messages`);
                }
            }
            if (lid in fr) {
                let id = fr[lid];
                if (this.lidToLabel.has(id)) {
                    const label = this.lidToLabel.get(id);
                    ms = Array.from(this.midToMessage.values())
                        .filter(message => message.labels.has(label))
                        .slice(fr["first"], fr["last"]);
                } else {
                    console.log(`Cannot find lid ${id} for messages`);
                }
            }
            result.push({
                "header": {
                    "error": OK,
                    "md5": this.md5,
                    "countTotal": ms.length,
                    "countUnread": ms.filter(message => message.status.has(MessageStatus.UNREAD)).length,
                    "modified": true,
                    "batchCount": 1
                },
                "messageBatch": {
                    "messages": ms.map((msg) => Mailbox.serializeMessage(msg, fr["threaded"]))
                }
            });
        }
        return result;
    }

    private body(request: Request): any[] {
        let result: any = [];
        for(let mid of request.body["mids"].split(",")) {
            if (this.midToMessage.has(mid)) {
                result.push(Mailbox.serializeMessageBody(this.midToMessage.get(mid)));
            } else {
                console.log(`Cannot find mid ${mid} for message_body`);
            }
        }
        return result;
    }

    private attachment(request: Request): object {
        const params = new URLSearchParams(request.url.replace(MobileAPI.ATTACH + "?", ""));
        const mid = params.get("mid") || request.body["mid"];
        const hid = params.get("hid") || request.body["hid"];
        const thumb = params.get("thumb");
        if (this.midToMessage.has(mid)) {
            const message = this.midToMessage.get(mid);
            if (message.attachments.has(hid)) {
                const url = message.attachments.get(hid).url;
                return {
                    "status": {
                        status: OK
                    },
                    "url": url.startsWith("http")? url: `http://${request.host}${url}?thumb=y&sid=1`
                }
            } else {
                console.log(`Cannot find attachment ${hid} in mid ${mid} for attachment`);
            }
        } else {
            console.log(`Cannot find mid ${mid} for attachment`);
        }
        return {
            "status": {
                "status": ERROR
            }
        }
    }

    response(request: Request, realResponse: any) {
        this.init();
        let handle = urlToMobileAPIHandler(request.url);
        switch (handle) {
            case MobileAPI.XLIST: return this.staticResponse[handle];
            case MobileAPI.MESSAGES: return this.messages(request);
            case MobileAPI.MESSAGE_BODY: return this.body(request);
            case MobileAPI.ATTACH: return this.attachment(request);
            default: return realResponse
        }
    }

}
