import * as url from "url";
import {PublicConfigurationsProvider} from "../configurations/registry";
import {IncomingMessage} from "http";
import {Utils} from "../utils/utils";
import {ConfigurationKey, Router} from "./router";

export class PathConfigurationParser {
    private static X_PROXY_DOMAIN = 'xp.yandex-team.ru';

    public parse(req: IncomingMessage): ConfigurationKey {
        let forwardHost = this.parseForwardHost(req);
        let configurationName = this.parseConfigurationName(req);
        var parameters = null;
        if (configurationName) {
            let configuration = PublicConfigurationsProvider.demandByName(configurationName);
            let parametersCount = configuration.length - 1;
            if (Utils.isAsync(configuration)) {
                parametersCount -= 1;
            }
            parameters = this.parseParameters(req, configurationName, parametersCount);
            if (parameters.length != parametersCount) {
                throw Error(`Configuration '${configurationName}' requires at least ${parametersCount} parameters in path`);
            }
        }
        return new ConfigurationKey(forwardHost, configurationName, parameters);
    }

    public forwardedUrl(req: IncomingMessage, key: ConfigurationKey): string {
        const parts = this.parts(req);
        var i = 0;
        var result = req.url;
        while (i < parts.length) {
            if (parts[i] === 'f') {
                result = result.replace(`/f/${parts[i + 1]}`, '');
                i += 2;
            } else if (parts[i] === 'c') {
                var proxyCommand = `/c/${key.configurationName}`;
                if (key.parameters && key.parameters.length > 0) {
                    proxyCommand += '/' + key.parameters.join('/')
                }
                result = result.replace(proxyCommand, '');
                i += 2 + key.parameters.length;
            } else {
                i += 1
            }
        }
        return result;
    }

    private parseForwardHost(req: IncomingMessage): string | null {
        let parts = this.parts(req);
        for (let i = 0; i < parts.length; i++) {
            if (parts[i] == 'f') {
                return 'https://' + parts[i + 1];
            }
        }
        let host = req.headers.host;
        let schema = req.headers['x-forwarded-proto'];
        if (host && Utils.endsWith(host, `.${PathConfigurationParser.X_PROXY_DOMAIN}`)) {
            let forwardHost = host
                .replace(`.${PathConfigurationParser.X_PROXY_DOMAIN}`, '')
                .replace(/_/gi, '.');
            return `${schema || 'https'}://${forwardHost}`;
        }
        return null;
    }

    private parseConfigurationName(req: IncomingMessage): string | undefined {
        let parts = this.parts(req);
        for (let i = 0; i < parts.length; i++) {
            if (parts[i] == 'c') {
                return parts[i + 1];
            }
        }
        return null;
    }

    private parseParameters(req: IncomingMessage, name: string, count: number): string[] {
        let parts = this.parts(req);
        for (var i = 0; i < parts.length; i++) {
            if (parts[i] == 'c') {
                let firstParameter = i + 2;
                return parts.slice(firstParameter, firstParameter + count);
            }
        }
        return [];
    }

    private parts(req: IncomingMessage): string[] {
        return url.parse(req.url).pathname.split('/').slice(1);
    }
}
