import {IncomingMessage, ServerResponse} from "http";
import {MANDATORY_CONFIGURATIONS} from "../configurations/registry";
import * as toxy from 'toxy';
import * as rocky from 'rocky';
import {PublicConfigurationsProvider} from "../configurations/registry";
import {Utils} from "../utils/utils";
import {LoggingPoisonProvider} from "../poisons/logger";
import {PathConfigurationParser} from "./path-configuration-parser";

export class Router {
    private static ADMIN_PORT = 9000;

    private rocky;
    private runningConfigurations: RunningConfiguration[] = [];
    private admin;
    private history: string[] = [Router.timed('Started')];

    constructor(private config: any) {
        this.rocky = rocky();
        this.admin = toxy.admin({cors: true});
    }

    public start(port: number) {
        this.admin.listen(Router.ADMIN_PORT);
        console.log('Admin server listening on port', Router.ADMIN_PORT);

        this.rocky.get('/').use((req: IncomingMessage, res: ServerResponse, next: () => void) => {
            res.setHeader('content-type', 'text/html; charset=utf-8');
            res.write(this.history.join('\n'));
            res.end();
            next();
        });
        this.rocky.get('/favicon.ico').use();
        for (const pathPrefix in this.config.pathConfigurations) {
            const key = this.config.pathConfigurations[pathPrefix];
            this.rocky.all(pathPrefix).use((req, res, next) => {
                this.route(req, res, next, key);
            });
        }
        this.rocky.all('/*').use((req: IncomingMessage, res: ServerResponse, next: () => void) => {
            this.route(req, res, next, null);
        });
        this.rocky.listen(port);

        console.log('Incoming requests listening on port', port);
    }

    public stop() {
        console.log('Stopping router');
        this.admin.close(() => {});
        this.rocky.close();
    }

    private route(req: IncomingMessage, res: ServerResponse, next: () => void, requiredConfiguration: ConfigurationKey | null): void {
        try {
            this.routeImpl(req, res, next, requiredConfiguration);
        } catch (e) {
            console.error('Error on routing ' + req.url, e);
            throw e;
        }
    }

    private routeImpl(req: IncomingMessage, res: ServerResponse, next: () => void, requiredConfiguration: ConfigurationKey | null): void {
        console.log('Routing request:', req.url);
        // @ts-ignore
        this.history.unshift(Router.timed(LoggingPoisonProvider.curl(req, req.body)));

        let cb = (configuration: RunningConfiguration) => {
            configuration.modifyRequest(req);
            next();
        };

        let key = requiredConfiguration || new PathConfigurationParser().parse(req);
        let suitableConfiguration = this.runningConfigurations.filter(configuration => configuration.matches(key))[0];
        if (suitableConfiguration) {
            cb(suitableConfiguration);
            return;
        }

        Utils.randomFreePort(freePort => {
            let newConfiguration = new RunningConfiguration(key, freePort, this.config.defaultForwardHost);
            let cb2 = (error) => {
                if (error) {
                    console.error(`Cant create configuration, cause: ${error}`);
                    res.writeHead(500, error.message);
                    res.end();
                    return;
                }
                this.runningConfigurations.push(newConfiguration);
                cb(newConfiguration);
            };
            newConfiguration.run(this.admin, cb2);
        });
    }

    private static timed(text: string): string {
        let moscowTime = new Date().toLocaleString('ru', { timeZone: 'Europe/Moscow' });
        return `<p><font color="blue">${moscowTime}</font>&nbsp;${text}</p>`
    }
}

export class ConfigurationKey {
    constructor(readonly forwardHost?: string, readonly configurationName?: string, readonly parameters?: string[]) {
    }
}

class RunningConfiguration {
    constructor(private key: ConfigurationKey, private toxyPort: number, private defaultForwardHost: string) {
    }

    public matches(other: ConfigurationKey): boolean {
        return this.key.forwardHost == other.forwardHost
            && this.key.configurationName == other.configurationName
            && JSON.stringify(this.key.parameters) == JSON.stringify(other.parameters);
    }

    public run(admin, cb): void {
        const forwardHost = this.key.forwardHost || this.defaultForwardHost;
        console.log(`Starting Toxy on port ${this.toxyPort} for configuration "${this.key.configurationName}" forwarding to ${forwardHost}`);
        var proxy = toxy({secure: false});
        proxy.forward(forwardHost);
        for (let configuration of MANDATORY_CONFIGURATIONS) {
            configuration.configure(proxy);
        }
        let cb2 = (error) => {
            proxy.all('/*');
            proxy.listen(this.toxyPort);
            admin.manage(proxy);
            cb(error);
        };
        if (this.key.configurationName) {
            let applyConfiguration = PublicConfigurationsProvider.demandByName(this.key.configurationName);
            var parameters = [proxy].concat(this.key.parameters);
            if (Utils.isAsync(applyConfiguration)) {
                parameters = parameters.concat([cb2]);
            }
            try {
                applyConfiguration.apply(null, parameters);
            } catch(e) {
                cb(e);
            }
            if (!Utils.isAsync(applyConfiguration)) {
                cb2(null);
            }
        } else {
            cb2(null);
        }
    }

    public modifyRequest(req: IncomingMessage) {
        req.url = new PathConfigurationParser().forwardedUrl(req, this.key);
        // @ts-ignore
        req.rocky.options.target = `http://localhost:${this.toxyPort}`;
    }
}
