import * as fs from 'fs';
import {PatchResponsePoisonProvider, ResponsePatcher} from '../../patch'
import {ExperimentConfiguration} from "../experiment";
import {MobileAPI} from "../handlers";
import {Toxy, ToxyConfiguration} from "../../../core/configuration";

export interface MessageTimeStrategy {
    getTimestamp(order: number): Date
}

export class BucketTimeStrategy implements MessageTimeStrategy {
    private readonly config: { [bucket: string]: number };
    private readonly start: Date = new Date();

    constructor() {
        this.config = JSON.parse(fs.readFileSync('poisons/mail/buckets/config.json', 'utf8'));
    }

    public getTimestamp(order: number): Date {
        let bucket = this.getBucket(order);
        let now = new Date(this.start.getTime() - 1000 * order); // older messages should be down in message list
        switch (bucket) {
            case "today":
                return now;
            case "yesterday":
                return this.addDays(now, -1);
            case "thisWeek":
                return this.addDays(now, -3);
            case "lastWeek":
                return this.addDays(now, -10);
            case "thisMonth":
                return this.addDays(now, -20);
            default:
                return this.addDays(now, -60)
        }
    }

    private getBucket(order: number): string {
        var total = 0;
        for (let bucket in this.config) {
            total += this.config[bucket];
            if (order < total) {
                return bucket;
            }
        }
        return "months";
    }

    private addDays(date: Date, days: number) {
        // @ts-ignore
        var result = new Date(date);
        result.setDate(result.getDate() + days);
        return result;
    }
}

export class MessagesTimeModifier implements ResponsePatcher {
    cache: { [timestamp: number]: number } = {};
    private readonly strategy: MessageTimeStrategy;

    constructor(strategy: MessageTimeStrategy) {
        this.strategy = strategy
    }

    patch(jsonBody: any): any {
        let messages = jsonBody[0]['messageBatch']['messages'];
        if (!messages) {
            return jsonBody;
        }
        let timestamps: number[] = [];
        for (let timestamp in this.cache) {
            // @ts-ignore
            timestamps.push(timestamp);
        }
        for (let message of messages) {
            timestamps.push(parseInt(message['timestamp']));
        }
        timestamps.sort().reverse();
        for (var i = 0; i < timestamps.length; i++) {
            let timestamp = timestamps[i];
            if (!this.cache[timestamp]) {
                this.cache[timestamp] = this.strategy.getTimestamp(i).getTime()
            }
        }
        for (let message of messages) {
            let oldTimestamp = message['timestamp'];
            let newTimestamp = this.cache[oldTimestamp];
            message['timestamp'] = newTimestamp + "";
            message['utc_timestamp'] = Math.floor(newTimestamp / 1000) + "";
        }
        return jsonBody;
    }

}

export class TimeBucketConfiguration implements ToxyConfiguration {
    configure(toxy: Toxy) {
        new ExperimentConfiguration("buckets.json").configure(toxy);

        let strategy = new BucketTimeStrategy();
        let patcher = new MessagesTimeModifier(strategy);
        let poison = new PatchResponsePoisonProvider(patcher).create();
        toxy.all(MobileAPI.MESSAGES).poison(poison);
    }
}
