import {IncomingHttpHeaders, IncomingMessage, ServerResponse} from "http";
import {Poison, PoisonProvider} from "./poison";
import {parse} from "querystring";

export interface Request {
    readonly method: string;
    readonly url: string;
    readonly body?: string;
    readonly host: string;
    readonly headers: IncomingHttpHeaders;
}

export interface ResponsePatcher {
    patch(jsonBody: any, request?: Request): any
}

export class PatchResponsePoisonProvider implements PoisonProvider {
    private patcher: ResponsePatcher;

    constructor(patcher: ResponsePatcher) {
        this.patcher = patcher;
    }

    create(): Poison {
        return (req: IncomingMessage, res: ServerResponse, next: () => void) => {
            let _end = res.end;
            let _write = res.write;

            delete req.headers['accept-encoding'];

            let backendResponseBody: any[] = [];

            // @ts-ignore
            res.write = (chunk: any, encoding?: string, cb?: (error: Error | null | undefined) => void): boolean => {
                backendResponseBody.push(chunk);
                if (cb) {
                    cb(null);
                }
                return true
            };

            // @ts-ignore
            res.end = (chunk: any, encoding?: string, cb?: () => void): void => {
                if (chunk && typeof chunk !== 'function') {
                    backendResponseBody.push(chunk);
                }

                let body = Buffer.concat(backendResponseBody).toString();

                var modifiedBody = body;
                try {
                    let jsonBody = JSON.parse(body);

                    let parsedBody = PatchResponsePoisonProvider.parseRequestBody(req);
                    let modifiedJsonBody = this.patcher.patch(jsonBody, {
                        method: req.method,
                        url: req.url,
                        body: parsedBody,
                        host: req.headers.host,
                        headers: req.headers
                    });
                    modifiedBody = JSON.stringify(modifiedJsonBody);
                } catch (e) {
                    console.error(e);
                }

                res.end = _end;
                res.write = _write;
                res.removeHeader('content-length');
                res.end(modifiedBody, encoding, cb);
            };

            next();
        };
    }

    private static parseRequestBody(req: IncomingMessage): any {
        // @ts-ignore
        let requestBody = req.body;
        if (requestBody == null) {
            return null;
        }
        let contentType = req.headers['content-type'];
        if (contentType && contentType.indexOf('multipart') >= 0) {
            return requestBody;
        }
        let requestBodyString = requestBody.toString();
        if (contentType && contentType.indexOf('json') >= 0) {
            return JSON.parse(requestBodyString);
        }
        return parse(requestBodyString);
    }
}
