import {IncomingMessage, ServerResponse} from "http";
import * as toxy from 'toxy';
import * as fs from "fs";
import * as Jimp from "jimp";
import * as path from "path";

export interface Poison {
    (req: IncomingMessage, res: ServerResponse, next: () => void): void
}

export class Poisons {
    /**
     * Infects the HTTP flow injecting a latency jitter in the response
     * @see https://github.com/h2non/toxy#latency
     *
     * @param jitter - Jitter value in milliseconds
     * @param min - Random jitter maximum value
     * @param max - Random jitter minimum value
     */
    static latency(jitter?: number, min?: number, max?: number) {
        return toxy.poisons.latency({
            jitter: jitter,
            min: min,
            max: max
        })
    }

    /**
     * Aborts the TCP connection. From the low-level perspective, this will destroy the socket on the server,
     * operating only at TCP level without sending any specific HTTP application level data.
     * @see https://github.com/h2non/toxy#abort-connection
     *
     * @param delay - Aborts TCP connection after waiting the given milliseconds
     * @param next - If true, the connection will be aborted if the target server takes more than the delay param time to reply
     * @param error - Custom internal node.js error to use when destroying the socket
     */
    static abort(delay: number = 0, next: boolean = false, error: Error = null): Poison {
        return toxy.poisons.abort()
    }

    static responseBy(code: number, body: string, headers?: { [name: string]: string }, encoding?: string): Poison {
        return toxy.poisons.inject({
            code: code,
            body: body,
            headers: headers,
            encoding: encoding
        });
    }

    static responseByImage(file: string, width: number, height: number, color: string, cb) {
        new Jimp(width, height, "#" + color, (err, image) => {
            if (err) {
                cb(null, err);
                return;
            }
            let mime = null;
            switch (path.extname(file)) {
                case "png": {
                    mime = Jimp.MIME_PNG;
                    break;
                }
                // case "gif": {
                //     mime = Jimp.MIME_GIF;
                //     break;
                // }
                case "jpeg": {
                    mime = Jimp.MIME_JPEG;
                    break;
                }
                default: {
                    mime = Jimp.MIME_PNG;
                }
            }
            image.getBufferAsync(mime).then((buffer) => {
                let poison = toxy.poisons.inject({
                    code: 200,
                    body: buffer,
                    headers: {"Content-Disposition": `inline; filename="${file}"`}
                });
                cb(poison, null);
            });
        });
    }

    /**
     * Limits the amount of bytes sent over the network in outgoing HTTP traffic for a specific time frame.
     * @see https://github.com/h2non/toxy#bandwidth
     *
     * @param bytes - Amount of chunk of bytes to send. Default 1024
     * @param threshold - Packets time frame in milliseconds. Default 1000
     */
    static bandWidth(bytes?: number, threshold?: number): Poison {
        return toxy.poisons.bandwidth({
            bytes: bytes,
            threshold: threshold
        });
    }

    /**
     * Delays the HTTP connection ready state.
     * @see https://github.com/h2non/toxy#slow-open
     *
     * @param delay - Delay connection in milliseconds
     */
    static responseWithDelay(delay: number): Poison {
        return toxy.poisons.slowOpen({
            delay: delay
        });
    }

    static infinitelyLongResponse(): Poison {
        return this.responseWithDelay(1_000_000_000);
    }

    /**
     * Delays the HTTP connection close signal (EOF).
     * @see https://github.com/h2non/toxy#slow-close
     *
     * @param delay - Delay time in milliseconds
     */
    static delayClose(delay: number): Poison {
        return toxy.poisons.slowClose({
            delay: delay
        });
    }

    /**
     * Limits the amount of requests received by the proxy in a specific threshold time frame.
     * Designed to test API limits. Exposes typical X-RateLimit-* headers.
     *
     * @param limit - Total amount of requests
     * @param threshold - Limit time frame in milliseconds
     * @param code - HTTP status code when limit is reached
     * @param message - Optional error message when limit is reached
     */
    static rateLimit(limit: number = 10, threshold: number = 1000, code: number = 429, message?: string) {
        // noinspection TypeScriptValidateJSTypes
        return toxy.poisons.rateLimit({
            limit: limit,
            threshold: threshold,
            code: code,
            message: message
        });
    }
    static responseJsonFrom(path: string): Poison {
        let config = JSON.parse(fs.readFileSync(path, 'utf8'));
        return Poisons.responseBy(
            config.code,
            JSON.stringify(config.body),
            config.headers,
            config.encoding
        )
    }

    static responseJson(code: number, body: any): Poison {
        return this.responseBy(code, JSON.stringify(body))
    }

    static responseError500(): Poison {
        return this.responseJsonFrom('resources/mail/error/500.json');
    }

    static responseServiceUnavailable(): Poison {
        return Poisons.responseBy(503, 'Service Unavailable')
    }

    static responseError200(): Poison {
        return this.responseJsonFrom('resources/mail/error/auth.json');
    }
}

export interface PoisonProvider {
    create(): Poison
}
