import Foundation

typealias FolderId = Int64
typealias LabelId = String
typealias MessageId = Int64
typealias ThreadId = Int64

let apiBase = "https://mail.yandex.ru/api/mobile"

enum FolderType: UInt8 {
    case inbox = 1,
    user,
    outbox,
    sent,
    draft,
    spam,
    trash,
    archive,
    templates,
    discounts,
    unknown,
    unsubscribe
}

protocol Folder {
    var id: FolderId {get}
    var name: String {get}
    var type: FolderType {get}
    var parent: FolderId? {get}
    var count: UInt32 {get}
    var unread: UInt32 {get}
}

enum LabelType: UInt8 {
    case user = 1,
    social,
    system
}

protocol Label {
    var id: LabelId {get}
    var name: String {get}
    var type: LabelType {get}
    var color: String {get}
    var count: UInt32 {get}
    var unread: UInt32 {get}
}

struct Address: CustomStringConvertible, Hashable {

    let email: String
    let name: String?

    init?(email: String, name: String? = nil) {
        guard email.contains("@") else {
            return nil
        }
        self.email = email
        self.name = name
    }

    var description: String {
        if let name = self.name {
            return "\(name)<\(self.email)>"
        }
        return "\(self.email)"
    }
}

protocol Message {
    var id: MessageId {get}
    var parent: FolderId {get}
    var thread: ThreadId? {get}
    var labels: Set<LabelId> {get}
    var subject: String {get}
    var from: Address {get}
    var to: Set<Address> {get}
    var cc: Set<Address> {get}
    var bcc: Set<Address> {get}
    var firstLine: String {get}
}

protocol Mailbox {
    func folderById(_ fid: FolderId) -> Folder?;
    func labelById(_ lid: LabelId) -> Label?;
    func folderIds() -> [FolderId];
    func labelIds() -> [LabelId];
    func messagesIds() -> [MessageId];
    func messageById(_ mid: MessageId) -> Message?;
    func messageIdByFolder(_ fid: FolderId) -> [MessageId]
}

class FolderModel: Folder, CustomStringConvertible {

    let id: FolderId
    let name: String
    let type: FolderType
    let parent: FolderId?
    let count: UInt32
    let unread: UInt32

    init(fid: FolderId, name: String, type: UInt8, parent: FolderId?, count: UInt32 = 0, unread: UInt32 = 0) {
        self.id = fid
        self.name = name
        self.type = FolderType(rawValue: type) ?? .unknown
        self.parent = parent
        self.count = count
        self.unread = unread
    }

    var description: String {
        return "Folder(fid: \(self.id), name: \(self.name), type: \(self.type))"
    }
}

class LabelModel: Label, CustomStringConvertible {

    let id: LabelId
    let name: String
    let type: LabelType
    let color: String
    let count: UInt32
    let unread: UInt32

    init(lid: LabelId, name: String, type: UInt8, color: String = "FFFFFF", count: UInt32 = 0, unread: UInt32 = 0) {
        self.id = lid
        self.name = name
        self.type = LabelType(rawValue: type) ?? .user
        self.color = color
        self.count = count
        self.unread = unread
    }

    var description: String {
        return "Folder(fid: \(self.id), name: \(self.name), type: \(self.type))"
    }
}

class MessageModel: Message, CustomStringConvertible {

    let id: MessageId
    let parent: FolderId
    let thread: ThreadId?
    let labels: Set<LabelId>
    let subject: String
    let from: Address
    let to: Set<Address>
    let cc: Set<Address>
    let bcc: Set<Address>
    let firstLine: String

    init?(id: MessageId,
         parent: FolderId,
         subject: String,
         from: Address,
         thread: ThreadId? = nil,
         labels: [LabelId] = [],
         to: [Address] = [],
         cc: [Address] = [],
         bcc: [Address] = [],
         firstLine: String? = nil
        ) {
        self.id = id
        self.parent = parent
        self.thread = thread
        self.labels = Set(labels)
        self.subject = subject
        self.from = from
        self.to = Set(to)
        self.cc = Set(to)
        self.bcc = Set(to)
        self.firstLine = firstLine ?? ""
//        if self.to.isEmpty && self.cc.isEmpty && self.bcc.isEmpty {
//            return nil
//        }
    }

    var description: String {
        return "Message(mid: \(self.id), subject: \(self.subject), from: \(self.from))"
    }

}

class MailboxModel: Mailbox {

    private var folders: [FolderId:Folder] = [:]
    private var labels: [LabelId:Label] = [:]
    private var messages: [FolderId: [Message]] = [:]
    private var messageById: [MessageId:Message] = [:]

    init(folders: [Folder], labels: [Label], messages: [FolderId: [Message]]) {
        for folder in folders {
            self.folders[folder.id] = folder
        }
        for label in labels {
            self.labels[label.id] = label
        }
        self.messages = messages
        for folder in self.messages.values {
            for message in folder {
                self.messageById[message.id] = message
            }
        }
    }

    func folderIds() -> [FolderId] {
        return Array(folders.keys)
    }

    func labelIds() -> [LabelId] {
        return Array(labels.keys)
    }

    func folderById(_ fid: FolderId) -> Folder? {
        return folders[fid]
    }

    func labelById(_ lid: LabelId) -> Label? {
        return labels[lid]
    }

    func messagesIds() -> [MessageId] {
       return Array(messageById.keys)
    }

    func messageById(_ mid: MessageId) -> Message? {
            return messageById[mid]
    }

    func messageIdByFolder(_ fid: FolderId) -> [MessageId] {
        return messages[fid]?.map({$0.id}) ?? []
    }

}

fileprivate func request(url: URL, requestBody: Data? = nil, token: String) -> Any {
    let semaphore = DispatchSemaphore(value: 0);
    var request = URLRequest(url: url)
    if requestBody != nil {
        request.httpMethod = "POST"
        request.httpBody = requestBody
        request.addValue("application/json", forHTTPHeaderField: "Content-Type")
    }

    var result: Any = ""

    request.setValue("OAuth \(token)", forHTTPHeaderField: "Authorization")
    let task = URLSession.shared.dataTask(with: request) {(data, response, error) in
        guard let data = data else {
            return
        }
        result = try! JSONSerialization.jsonObject(with: data, options: .mutableContainers)
        semaphore.signal()
    }

    task.resume()
    semaphore.wait()

    return result
}

fileprivate func parseXList(token: String) -> (folders: [Folder], labels: [Label]) {
    var folders: [Folder] = []
    var labels: [Label] = []
    let elements = request(url: URL(string: "\(apiBase)/v1/xlist")!, token: token) as! [Any]
    for element in elements {
        let json = element as! [String:Any]
        if let fid = json["fid"] as? String {
            folders.append(FolderModel(
                fid: FolderId(fid)!,
                name: json["display_name"] as! String,
                type: json["type"] as! UInt8,
                parent: json["parent"] as? FolderId,
                count: json["count_all"] as! UInt32,
                unread: json["count_unread"] as! UInt32
            ))
        }
        if let lid = json["lid"] as? String {
            labels.append(LabelModel(
                lid: LabelId(lid),
                name: json["display_name"] as! String,
                type: json["type"] as! UInt8,
                color: json["color"] as! String,
                count: json["count_all"] as! UInt32,
                unread: json["count_unread"] as! UInt32
            ))
        }
    }
    return (folders, labels)
}

fileprivate func parseMessages(folders: [Folder], count: UInt16 = 20, threaded: Bool = false, token: String) -> [FolderId: [Message]] {
    var result: [FolderId: [Message]] = [:]
    var json = [String: Any]()
    let parameters = folders.map {["fid": String($0.id), "threaded": threaded, "first": 0, "last": count, "md5": "", "returnIfModified": true] }
    json["requests"] = parameters
    let data = try? JSONSerialization.data(withJSONObject: json, options: [])
    let elements = request(url: URL(string: "\(apiBase)/v1/messages")!, requestBody: data, token: token) as! [Any]
    let dict2Address = { (dict: [String: Any]) -> Address in Address(email: dict["email"] as! String, name: dict["name"] as? String)! }
    for element in elements {
        let json = element as! [String: Any]
        if let batch = json["messageBatch"] as? [String: Any], let messages = batch["messages"] as? [[String: Any]] {
            for message in messages {
                let fid = FolderId(message["fid"] as! String)!
                if result[fid] == nil {
                    result[fid] = [Message]()
                }
                result[fid]!.append(MessageModel(
                    id: MessageId(message["mid"] as! String)!,
                    parent: fid,
                    subject: message["subjText"] as! String,
                    from: dict2Address(message["from"] as! [String: Any]),
                    thread: message["tid"] as? ThreadId,
                    labels: message["lid"] as! [LabelId],
                    to: (message["recipients"] as? [[String: Any]] ?? [["type": -1]]).filter({$0["type"] as! Int == 1}).map(dict2Address),
                    cc: (message["recipients"] as? [[String: Any]] ?? [["type": -1]]).filter({$0["type"] as! Int == 3}).map(dict2Address),
                    bcc: (message["recipients"] as? [[String: Any]] ?? [["type": -1]]).filter({$0["type"] as! Int == 4}).map(dict2Address),
                    firstLine: message["firstLine"] as? String
                )!)

            }
        }
    }
    return result
}

extension MailboxModel {
    static func create(forUser token: String) -> Mailbox {
        let (folders, labels) = parseXList(token: token)
        let messages = parseMessages(folders: folders, token: token)
        return MailboxModel(folders: folders, labels: labels, messages: messages)
    }
}

let token = "XXX"

print(MailboxModel.create(forUser: token).messageIdByFolder(1))


