#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys

sys.path.append("/usr/share/xeno/")
import yaml
import json
from collections import defaultdict
from time import sleep
from xeno_monitor import load_url


def add_bucket_url(node, bucket_id):
    return "http://{}:8080/add_bucket?bucket_id={}".format(node, bucket_id)


def add_shards_to_bucket_url(node, bucket_id, shard_ids):
    return "http://{}:8080/add_shards_to_bucket?bucket_id={}&shard_ids={}".format(
        node, bucket_id, ",".join(shard_ids)
    )


def del_shards_from_bucket_url(node, bucket_id, shard_ids):
    return "http://{}:8080/del_shards_from_bucket?bucket_id={}&shard_ids={}".format(
        node, bucket_id, ",".join(shard_ids)
    )


def acquired_buckets_info_url(node):
    return "http://{}:8080/acquired_buckets_info".format(node)


def get_current_state(nodes):
    state = {}
    state["shards_buckets_map"] = defaultdict(str)
    state["buckets_nodes_map"] = defaultdict(str)
    state["nodes_users_count"] = defaultdict(int)
    state["shards_users_count"] = defaultdict(int)
    for node in nodes:
        acquired_buckets = json.loads(load_url(acquired_buckets_info_url(node)))
        for bucket in acquired_buckets:
            for shard in bucket["shards"]:
                state["nodes_users_count"][node] += int(shard["users_count"])
                state["shards_users_count"][shard["id"]] = int(shard["users_count"])
                state["shards_buckets_map"][shard["id"]] = bucket["bucket_id"]
                state["buckets_nodes_map"][bucket["bucket_id"]] = node
    return state


def find_shard_to_move(buckets, state):
    shard_id, old_bucket, new_bucket, users_count = None, None, None, 0
    shards_in_right_bucket_count = 0
    shards_in_wrong_bucket_count = 0
    for bucket in buckets:
        for shard in bucket["shards"]:
            if state["shards_buckets_map"][shard] == bucket["name"]:
                shards_in_right_bucket_count += 1
                continue
            shards_in_wrong_bucket_count += 1
            node = state["buckets_nodes_map"][bucket["name"]]
            count = state["nodes_users_count"][node] + state["shards_users_count"][shard]
            if not shard_id or count < users_count:
                shard_id = shard
                old_bucket = state["shards_buckets_map"][shard]
                new_bucket = bucket["name"]
                users_count = count
    print (
        "shards in right bucket: {}, shard in wrong bucket: {}".format(
            shards_in_right_bucket_count, shards_in_wrong_bucket_count
        )
    )
    return shard_id, old_bucket, new_bucket


def main():
    if len(sys.argv) < 3:
        print "Usage {} <buckets_config_path> <nodes_list_path>".format(sys.argv[0])
        return
    buckets = yaml.load(open(sys.argv[1], "r"))["config"]["buckets"]
    nodes = [node.strip() for node in open(sys.argv[2], "r")]
    print ("buckets count: {}".format(len(buckets)))
    print ("nodes count: {}".format(len(nodes)))

    print ("add new buckets to nodes...")
    for bucket in buckets:
        for node in nodes:
            load_url(
                add_bucket_url(node, bucket["name"])
            )  # nothing happens if bucket already added
    sleep(10)

    while True:
        try:
            print ("getting current state...")
            state = get_current_state(nodes)
            print ("state: {}".format(json.dumps(state, sort_keys=True, indent=4)))
            shard_id, old_bucket, new_bucket = find_shard_to_move(buckets, state)
            if not shard_id:
                print ("nothing to do")
                sleep(10)
                continue
            print ("move shard {} from bucket {} to {}".format(shard_id, old_bucket, new_bucket))
            for node in nodes:
                load_url(del_shards_from_bucket_url(node, old_bucket, [shard_id]))
                load_url(add_shards_to_bucket_url(node, new_bucket, [shard_id]))
        except Exception as e:
            print (e)
        sleep(30)


if __name__ == "__main__":
    main()
