#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys

sys.path.append("/usr/share/xeno/")
from xeno_monitor import *
from threading import Thread, Lock
import logging
import json

INVALIDATE_AUTH_URL = "http://localhost:8080/invalidate_auth?uid={uid}"
SELECT_MAILISH_WITH_AUTH_BY_UIDS_QUERY = (
    "SELECT DISTINCT uid FROM mailish.auth_data WHERE uid IN ({})"
)
LOG_FILE = "/var/log/xeno/invalidate_auth.log"

auth_removed_count = 0
remove_failed_count = 0
lock = Lock()


def main():
    if len(sys.argv) != 3:
        print("usage {} <uids_path> <concurrency>".format(sys.argv[0]))
        sys.exit(1)
    uids_path = sys.argv[1]
    concurrency = int(sys.argv[2])
    setup_log()
    uids = read_uids(uids_path)
    invalidate_auth_with_concurrency(uids, concurrency)


def setup_log():
    logging.basicConfig(
        filename=LOG_FILE,
        filemode="a",
        format="[%(asctime)s] %(threadName)s %(levelname)s: %(message)s",
        level=logging.INFO,
        datefmt="%Y-%b-%d %H:%M:%S",
    )


def read_uids(filename):
    ret = []
    with open(filename, "r") as file:
        for line in file.readlines():
            uid = line.strip()
            ret.append(uid)
    return ret


def invalidate_auth_with_concurrency(uids, concurrency):
    shards = get_sharpei_stat().keys()
    chunk_size = len(shards) / concurrency
    threads = []
    while len(shards) > 0:
        chunk = shards[:chunk_size]
        shards = shards[chunk_size:]
        thread = Thread(
            target=invalidate_auth_from_shards,
            args=(
                chunk,
                uids,
            ),
        )
        threads.append(thread)
    logging.info("start removing: concurrency={}".format(len(threads)))
    for thread in threads:
        thread.start()
    logging.info("wait...")
    for thread in threads:
        thread.join()
    logging.info(
        "removing completed: removed={} failed={}".format(auth_removed_count, remove_failed_count)
    )


def invalidate_auth_from_shards(shards, uids):
    for shard_id in shards:
        invalidate_auth_from_shard(shard_id, uids)


def invalidate_auth_from_shard(shard_id, uids):
    logging.info("processing shard {}...".format(shard_id))
    if len(uids) == 0:
        return
    users = select_users_with_auth_data(shard_id, uids)
    for uid in users:
        logging.info("invalidate auth for uid={}".format(uid))
        try:
            invalidate_auth(uid)
            update_stats_on_remove_success()
        except Exception as e:
            logging.error("invalidate auth error: uid={}, msg={}".format(uid, str(e)))
            update_stats_on_remove_error()


def select_users_with_auth_data(shard_id, uids):
    cur = get_db_cursor(shard_id)
    users = fetch(cur, SELECT_MAILISH_WITH_AUTH_BY_UIDS_QUERY.format(",".join(uids)))
    return [user["uid"] for user in users]


@retry()
def invalidate_auth(uid):
    load_url(INVALIDATE_AUTH_URL.format(uid=uid))


def update_stats_on_remove_success():
    with lock:
        global auth_removed_count
        auth_removed_count += 1


def update_stats_on_remove_error():
    with lock:
        global remove_failed_count
        remove_failed_count += 1


if __name__ == "__main__":
    main()
