#pragma once

#include <common/account.h>
#include <common/errors.h>
#include <common/http.h>
#include <common/context.h>
#include <common/json.h>
#include <common/karma.h>

#include <ymod_httpclient/cluster_client.h>
#include <yplatform/util/safe_call.h>

#include <memory>
#include <string>

namespace xeno::auth {

struct auth_response
{
    std::string xtoken_id;
    std::string token_id;
    uid_t uid;
    karma_t karma;
    std::string user_ticket;
};
using auth_response_ptr = std::shared_ptr<auth_response>;
using auth_cb = std::function<void(error, auth_response_ptr)>;

inline void authorize(
    context_ptr task_ctx,
    const std::string& xtoken,
    const std::string& ip,
    unsigned port,
    const auth_cb& cb)
{
    using namespace std::string_literals;

    std::string url = "/blackbox?method=oauth";
    std::stringstream body;
    body << "format=json"
         << "&get_user_ticket=yes"
         << "&oauth_token=" << xtoken << "&userip=" << ip << "&user_port=" << port;

    auto request = http::request_t::POST(url, body.str());
    auto client = yplatform::find<yhttp::cluster_client>("blackbox_client");
    client->async_run(
        task_ctx, std::move(request), [cb, task_ctx](error err, yhttp::response response) {
            try
            {
                if (err)
                {
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "blackbox oauth error: " << err.message();
                    return yplatform::safe_call(task_ctx, "", cb, err, auth_response_ptr());
                }

                if (response.status != 200)
                {
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "blackbox oauth error: bad status: " << response.status << " "
                        << response.reason;
                    return yplatform::safe_call(
                        task_ctx,
                        "",
                        cb,
                        error(code::auth_error, response.body),
                        auth_response_ptr());
                }

                json::value result;
                json::reader reader;
                if (!reader.parse(response.body, result))
                {
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "blackbox oauth error: bad json";
                    return yplatform::safe_call(
                        task_ctx,
                        "",
                        cb,
                        error(code::auth_error, "invalid response: "s + response.body),
                        auth_response_ptr());
                }

                auto status = result["status"];
                auto err_id = status["id"].asInt();
                if (err_id)
                {
                    auto message = result["error"];
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "blackbox oauth error: "
                        << (message.isNull() ? "unknown error" : message.asString());
                    yplatform::safe_call(
                        task_ctx,
                        "",
                        cb,
                        error(code::auth_error, response.body),
                        auth_response_ptr());
                }
                else
                {
                    auto response = std::make_shared<auth_response>();
                    auto oauth = result["oauth"];
                    response->token_id = oauth["token_id"].asString();
                    response->xtoken_id = oauth.get("xtoken_id", "").asString();
                    if (response->xtoken_id.empty())
                    {
                        response->xtoken_id = response->token_id;
                    }
                    response->uid = std::stoull(oauth["uid"].asString());
                    response->karma.value = result["karma"]["value"].asInt();
                    response->karma.status = result["karma_status"]["value"].asInt();
                    response->user_ticket = result["user_ticket"].asString();
                    yplatform::safe_call(task_ctx, "", cb, err, response);
                }
            }
            catch (const std::exception& e)
            {
                YLOG_CTX(task_ctx->logger(), task_ctx, error)
                    << "blackbox oauth error: exception: " << e.what();
                yplatform::safe_call(
                    task_ctx, "", cb, error(code::auth_error, e.what()), auth_response_ptr());
            }
        });
}

}
