#pragma once

#include "social/get_access_token_op.h"
#include "social/cache.h"

#include <common/account.h>
#include <common/json.h>

#include <ymod_httpclient/cluster_client.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::auth {

struct get_access_token_op
{
    using yield_context = yplatform::yield_context<get_access_token_op>;
    using access_token_cb = social::access_token_cb;

    get_access_token_op(
        context_ptr task_ctx,
        const std::string& oauth_app,
        const std::string& refresh_token,
        social::social_settings_ptr settings,
        bool skip_cache_read,
        const access_token_cb& cb)
        : task_ctx(task_ctx)
        , oauth_app(oauth_app)
        , refresh_token(refresh_token)
        , settings(settings)
        , skip_cache_read(skip_cache_read)
        , cb(cb)
    {
    }

    void operator()(yield_context yield_ctx, error err = {})
    {
        try
        {
            reenter(yield_ctx)
            {
                if (!skip_cache_read)
                {
                    yield social::get_cached_token(task_ctx, oauth_app, refresh_token, yield_ctx);
                    if (!err)
                    {
                        yield break;
                    }
                    else
                    {
                        YLOG_CTX_LOCAL(task_ctx, error)
                            << "get token from cache error: " << err.message();
                    }
                }
                yield yplatform::spawn(std::make_shared<social::get_access_token_op>(
                    task_ctx, oauth_app, refresh_token, settings, yield_ctx));
                if (err)
                {
                    yield break;
                }
                cb_called = true;
                cb({}, access_token, expires_at);
                yield social::set_cached_token(
                    task_ctx, oauth_app, refresh_token, access_token, expires_at, yield_ctx);
                if (err)
                {
                    YLOG_CTX_LOCAL(task_ctx, error)
                        << "get token from cache error: " << err.message();
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(task_ctx, error) << "get_access_token_op exception: " << e.what();
            err = code::get_access_token_error;
        }
        if (yield_ctx.is_complete() && !cb_called)
        {
            cb(err, access_token, expires_at);
        }
    }

    void operator()(
        yield_context yield_ctx,
        error err,
        const std::string& access_token,
        const time_point& ts)
    {
        if (!err)
        {
            this->access_token = access_token;
            this->expires_at = ts;
        }
        (*this)(yield_ctx, err);
    }

    context_ptr task_ctx;
    std::string oauth_app;
    std::string refresh_token;
    social::social_settings_ptr settings;
    bool skip_cache_read = false;
    access_token_cb cb;
    std::string access_token;
    time_point expires_at;
    bool cb_called = false;
};

}

#include <yplatform/unyield.h>
