#pragma once

#include <common/account.h>
#include <common/http.h>
#include <common/context.h>
#include <common/errors.h>
#include <common/json.h>

#include <ymod_httpclient/cluster_client.h>
#include <yplatform/util/safe_call.h>

#include <string>
#include <map>

namespace xeno::auth {

struct get_or_create_mailish_response
{
    uid_t uid{ 0 };
    bool is_new_account{ false };
    std::string xtoken;
};

using get_or_create_mailish_cb = std::function<void(error, const get_or_create_mailish_response&)>;

inline void get_or_create_mailish(
    context_ptr task_ctx,
    const std::string& email,
    const std::string& mailish_id,
    const std::string& client_id,
    const std::string& client_secret,
    const std::multimap<string, string>& optional_args,
    const std::string& ip,
    const get_or_create_mailish_cb& cb)
{
    using namespace std::string_literals;

    static const std::vector<string> args{ "app_id",       "app_platform", "app_version",
                                           "manufacturer", "model",        "uuid",
                                           "deviceid",     "ifv",          "device_name",
                                           "device_id",    "code_verifier" };

    static const size_t mailish_id_max_size = 255;
    if (mailish_id.size() > mailish_id_max_size)
    {
        return yplatform::safe_call(
            task_ctx, cb, code::mailish_id_too_big, get_or_create_mailish_response());
    }

    std::string url = "/1/bundle/account/get_or_create/mailish/?consumer=mail";
    std::stringstream headers, body;
    headers << "Ya-Consumer-Client-Ip: " << ip << "\r\n";
    body << "email=" << email << "&client_id=" << client_id << "&client_secret=" << client_secret
         << "&mailish_id=" << mailish_id;

    for (auto& name : args)
    {
        auto it = optional_args.find(name);
        if (it != optional_args.end())
        {
            body << "&" << name << "=" << it->second;
        }
    }

    auto request = http::request_t::POST(url, headers.str(), body.str());
    auto client = yplatform::find<yhttp::cluster_client>("passport_internal_client");
    client->async_run(
        task_ctx, std::move(request), [cb, task_ctx](error err, yhttp::response response) mutable {
            try
            {
                if (err)
                {
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "get or create mailish error: " << err.message();
                    return yplatform::safe_call(
                        task_ctx, cb, err, get_or_create_mailish_response());
                }

                if (response.status != 200)
                {
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "get or create mailish error: bad status: " << response.status << " "
                        << response.reason;
                    return yplatform::safe_call(
                        task_ctx,
                        cb,
                        error(code::auth_error, response.body),
                        get_or_create_mailish_response());
                }

                json::value json;
                json::reader reader;
                if (!reader.parse(response.body, json))
                {
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "get or create mailish error: bad json";
                    return yplatform::safe_call(
                        task_ctx,
                        cb,
                        error(code::auth_error, "invalid response: "s + response.body),
                        get_or_create_mailish_response());
                }

                if (json["status"].asString() != "ok")
                {
                    std::vector<std::string> errors;
                    for (auto& error : json["errors"])
                    {
                        errors.push_back(error.asString());
                    }
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "get or create mailish error: " << boost::algorithm::join(errors, ", ");
                    return yplatform::safe_call(
                        task_ctx,
                        cb,
                        error(code::auth_error, response.body),
                        get_or_create_mailish_response());
                }

                get_or_create_mailish_response result;
                result.is_new_account = json["is_new_account"].asBool();
                result.xtoken = json["oauth_token"].asString();
                result.uid = std::stoull(json["uid"].asString());

                yplatform::safe_call(task_ctx, cb, error(), result);
            }
            catch (const std::exception& e)
            {
                YLOG_CTX(task_ctx->logger(), task_ctx, error)
                    << "get or create mailish error: exception: " << e.what();
                yplatform::safe_call(
                    task_ctx,
                    cb,
                    error(code::auth_error, e.what()),
                    get_or_create_mailish_response());
            }
        });
}

}
