#pragma once

#include <common/account.h>
#include <common/http.h>
#include <common/errors.h>
#include <common/context.h>
#include <common/json.h>
#include <common/karma.h>

#include <ymod_httpclient/cluster_client.h>
#include <yplatform/util/safe_call.h>

#include <algorithm>
#include <string>

namespace xeno::auth {

struct user_info
{
    karma_t karma;
};

using user_info_response = std::map<uid_t, user_info>;
using user_info_cb = std::function<void(error, const user_info_response&)>;

inline void get_user_info(
    context_ptr task_ctx,
    const uid_vector& uids,
    const std::string& ip,
    unsigned port,
    const user_info_cb& cb)
{
    using namespace std::string_literals;

    if (uids.empty())
    {
        return cb(code::ok, user_info_response());
    }

    std::string url = "/blackbox?method=userinfo";
    std::stringstream body;
    body << "format=json"
         << "&uid=";
    for (auto it = uids.begin(); it != uids.end(); ++it)
    {
        if (it != uids.begin()) body << ",";
        body << *it;
    }
    body << "&userip=" << ip << "&user_port=" << port;

    auto request = http::request_t::POST(url, body.str());
    auto client = yplatform::find<yhttp::cluster_client>("blackbox_client");
    client->async_run(
        task_ctx, std::move(request), [cb, task_ctx](error err, yhttp::response response) mutable {
            try
            {
                if (err)
                {
                    YLOG(task_ctx->logger(), error) << "get userinfo error: " << err.message();
                    return yplatform::safe_call(task_ctx, cb, err, user_info_response());
                }

                if (response.status != 200)
                {
                    YLOG(task_ctx->logger(), error)
                        << "get userinfo error: bad status: " << response.status << " "
                        << response.reason;
                    return yplatform::safe_call(
                        task_ctx, cb, error(code::auth_error, response.body), user_info_response());
                }

                json::value json;
                json::reader reader;
                if (!reader.parse(response.body, json))
                {
                    YLOG(task_ctx->logger(), error) << "get userinfo error: bad json";
                    return yplatform::safe_call(
                        task_ctx,
                        cb,
                        error(code::auth_error, "invalid response: "s + response.body),
                        user_info_response());
                }

                user_info_response result;
                auto users = json["users"];
                for (auto& user : users)
                {
                    uid_t uid = std::stoull(user["uid"]["value"].asString());
                    user_info info;
                    info.karma.value = user["karma"]["value"].asInt();
                    info.karma.status = user["karma_status"]["value"].asInt();
                    result.emplace(uid, std::move(info));
                }
                yplatform::safe_call(task_ctx, cb, code::ok, result);
            }
            catch (const std::exception& e)
            {
                YLOG(task_ctx->logger(), error) << "get userinfo error: exception: " << e.what();
                yplatform::safe_call(
                    task_ctx, cb, error(code::auth_error, e.what()), user_info_response());
            }
        });
}

}
