#pragma once

#include <common/json.h>
#include <common/types.h>

#include <ymod_httpclient/cluster_client.h>

namespace xeno::auth::social {

using cached_token_cb =
    std::function<void(error, const std::string& access_token, const time_point& expires_at)>;

inline void get_cached_token(
    context_ptr ctx,
    const std::string& oauth_app,
    const std::string& refresh_token,
    cached_token_cb cb)
{
    auto req = yhttp::request::POST(
        "/get", yhttp::form_encode({ { "key", oauth_app + "$" + refresh_token } }));

    auto rcache_client = yplatform::find<yhttp::cluster_client>("rcache_client");
    rcache_client->async_run(
        ctx, req, [cb = std::move(cb), ctx](error err, const yhttp::response& response) {
            if (err) return cb(err, {}, {});

            if (response.status == 200)
            {
                auto response_json = json::from_string(response.body);
                auto access_token = response_json["value"].asString();
                auto expires_at = time_traits::clock::now() +
                    time_traits::seconds(response_json["ttl"].asUInt64());
                return cb({}, access_token, expires_at);
            }
            else
            {
                YLOG_CTX(ctx->logger(), ctx, error)
                    << "get token from cache error: " << std::to_string(response.status) << " "
                    << response.reason;
                return cb(code::token_cacher_error, {}, {});
            }
        });
}

inline void set_cached_token(
    context_ptr ctx,
    const std::string& oauth_app,
    const std::string& refresh_token,
    const std::string& access_token,
    const time_point& expires_at,
    without_data_cb cb)
{
    auto now = time_traits::clock::now();
    if (now >= expires_at)
    {
        return;
    }
    auto ttl = time_traits::get_seconds_count(expires_at - now);
    auto req = yhttp::request::POST(
        "/set",
        yhttp::form_encode({ { "key", oauth_app + "$" + refresh_token },
                             { "value", access_token },
                             { "ttl", ttl } }));

    auto rcache_client = yplatform::find<yhttp::cluster_client>("rcache_client");
    rcache_client->async_run(
        ctx, req, [cb = std::move(cb), ctx](error err, const yhttp::response& response) {
            if (err) return cb(err);

            if (response.status / 100 == 4 || response.status / 100 == 5)
            {
                YLOG_CTX(ctx->logger(), ctx, error)
                    << "set token to cache error: " << std::to_string(response.status) << " "
                    << response.reason;
                return cb(code::token_cacher_error);
            }
            cb({});
        });
}

}
